(ns codescene.features.reports.portfolio.project-cards
  (:require
   [codescene.features.reports.common.pdf-elements :as pdf-elements]
   [codescene.features.reports.common.project-card :as project-card]
   [codescene.features.reports.common.styling :as styling]
   [codescene.features.reports.pdf-helpers :as ph]))

(defn- project-stats-part [project options]
  (ph/table [] [1 3 2 1 3 2]
            (ph/row
             (ph/cell [] "I")
             (ph/cell [] "# of commits:")
             (ph/cell [] "2350")
             (ph/cell [] "I")
             (ph/cell [] "Active developers:")
             (ph/cell [] "6"))))


(defn- bottom-left-part [project {:keys [img-dir] :as options}]
  (project-card/pdf-element-code-health (:renderable-trends project) img-dir))

(defn- unavailable-pr-stats [{:keys [pr-conf-url]} img-dir]
  (let [pr-img (styling/load-resource img-dir "portfolio/project-card/Pull request.png")]
    (ph/cell [:pc-rp-na-cell]
             (ph/image [:pc-rp-na-image] pr-img)
             (ph/table [:pc-rp-na-link-table] [20 23] (pdf-elements/link-row-small pr-conf-url "Integrate PRs" img-dir))
             (ph/paragraph [:pc-rp-na-text-paragraph]
                           "Integrate PRs on this project to see detected finding statistics and statuses."))))

(defn- available-pr-stats [{:keys [period-statistics links]} img-dir]
  (let [{:keys [pull-requests negative-findings findings-fixed findings-ignored findings-suppressed]} period-statistics
        {:keys [pr-stats-url]} links]
    (ph/cell [:pc-rp-cell]
             (ph/table [:pc-rp-link-table] [12 17] (pdf-elements/link-row-small pr-stats-url "PR statistics" img-dir))
             (ph/paragraph [:pc-rp-text-paragraph]
                           (ph/phrase [:pc-rp-text-phrase] "PR checks: ") (ph/phrase [:pc-rp-number-phrase] pull-requests))
             (ph/paragraph [:pc-rp-text-paragraph]
                           (ph/phrase [:pc-rp-text-phrase] "Findings detected: ")
                           (ph/phrase [:pc-rp-number-phrase] (if (> negative-findings 999) ">1k" negative-findings)))
             (ph/paragraph [:pc-rp-text-paragraph] (ph/phrase [:pc-rp-text-phrase] "Fixed: ")
                           (ph/phrase [:pc-rp-number-phrase] findings-fixed))
             (ph/paragraph [:pc-rp-text-paragraph] (ph/phrase [:pc-rp-text-phrase] "Ignored: ")
                           (ph/phrase [:pc-rp-number-phrase] findings-ignored))
             (ph/paragraph [:pc-rp-text-paragraph] (ph/phrase [:pc-rp-text-phrase] "Suppressed: ")
                           (ph/phrase [:pc-rp-number-phrase] findings-suppressed)))))

(defn- pr-stats-cell [{:keys [period-statistics links] :as project} {:keys [img-dir]}]
  (let [{:keys [pull-requests issues-detected issues-addressed issues-ignored]} period-statistics
        pr-stats-unavailable (every? styling/unavailable? [issues-detected issues-addressed issues-ignored pull-requests])]
    (if pr-stats-unavailable
      (unavailable-pr-stats links img-dir)
      (available-pr-stats project img-dir))))

(defn- title-paragraph [{:keys [name links]}]
  (let [url (:dashboard-url links)
        display-name (styling/string->cutoff-string name 43)]
    (ph/paragraph [:pc-title-phrase] (ph/anchor [] {:target url} display-name))))

(defn- icon-and-text-cells [img-dir icon-filename light-text heavy-text]
  (let [icon (styling/load-resource img-dir icon-filename)]
    [(ph/cell [:pc-rp-stats-cell] (ph/image [:pc-rp-stats-image] icon))
     (ph/cell [:pc-rp-stats-cell] (ph/paragraph []
                                                (ph/chunk [:pc-rp-light-chunk] light-text)
                                                (ph/chunk [:pc-rp-heavy-chunk] heavy-text)))]))


(defn- period-table [{:keys [beginning end]} {:keys [img-dir]}]
  (let [date-text (styling/interval->printable-string beginning end)]
    (ph/table [] [1 28]
              (apply ph/row
                     (icon-and-text-cells img-dir "portfolio/project-card/Calendar.png" "Statistics for: " date-text)))))

(defn- stats-table [{:keys [period-statistics group-name]} {:keys [img-dir]}]
  (let [{:keys [contributors commits]} period-statistics
        commits-text (styling/number->printable-string-100k commits)
        contributors-text (styling/number->printable-string-100k contributors)
        group-name-text (styling/string->cutoff-string group-name 18)]
    (ph/table [] [3 15 3 3 18 3 3 27]
              (apply ph/row
                     (concat
                      (icon-and-text-cells img-dir "portfolio/project-card/Git-commit.png" "# of commits: " commits-text)
                      [""]
                      (icon-and-text-cells img-dir "portfolio/project-card/Users.png" "Active developers: " contributors-text)
                      [""]
                      (if group-name
                        (icon-and-text-cells img-dir "portfolio/project-card/Group.png" "Group: " group-name-text)
                        ["" ""]))))))


(defn pdf-elements [projects {:keys [img-dir] :as options}]
  (for [{:keys [renderable-trends links] :as project} projects]
    (ph/table
     [:c-card-content-top-table] [25 8]
     (ph/row (ph/cell [:pc-left-cell]
                      (ph/table [] nil
                                (ph/row (ph/cell [:pc-title-cell] (title-paragraph project)))
                                (ph/row (ph/cell [:pc-stats-cell] (period-table project options)))
                                (ph/row (ph/cell [:pc-stats-cell] (stats-table project options)))
                                (ph/row (ph/cell [:pc-line-cell] (ph/line [:c-line])))
                                (ph/row (ph/cell [] (project-card/pdf-element-portfolio renderable-trends links img-dir)))))
             (pr-stats-cell project options)))))

