(ns codescene.features.reports.code-health.alerts
  (:require
    [codescene.features.reports.pdf-helpers :as ph]
    [codescene.features.reports.common.pdf-elements :as pdf-elements]
    [codescene.features.reports.common.styling :as styling]))


(defn get-top-n-alerts [alert-list n]
  (let [no-info-alert-list (remove #(= "info" (:category %)) alert-list)
        sorted-alert-list (sort-by #(.indexOf ["danger" "warning"] (:category %)) no-info-alert-list)]
    (take n sorted-alert-list)))

(defn- alert-contents [{:keys [img-dir http-base alerts-uri-fn]} column-widths {:keys [description linkable-name]}]
  (let [printable-description (styling/string->cutoff-string description 60)
        alerts-uri (alerts-uri-fn linkable-name)
        alerts-url (str http-base alerts-uri)
        description-paragraph (ph/paragraph [:tta-text-paragraph] printable-description)
        link-table (ph/table [] column-widths (pdf-elements/link-row alerts-url "View list" img-dir))]
    {:description-paragraph description-paragraph
     :link-table            link-table}))

(defn- alert-content-layout-single [{:keys [description-paragraph link-table]}]
  (ph/table
    [] nil
    (ph/row
      (ph/cell [:tta-single-text-cell] description-paragraph)
      (ph/cell [:tta-single-link-cell] link-table))))

(defn- alert-content-layout-multiple [{:keys [description-paragraph link-table]}]
  (ph/table
    [] nil
    (ph/row
      (ph/cell [:tta-text-cell] description-paragraph))
    (ph/row
      (ph/cell [:tta-link-cell] link-table))))

(defn- alert-red-cell [content]
  (ph/cell
    [:tta-alert-cell]
    (ph/table
      [] nil
      (ph/row
        (ph/cell [:c-leftborder-cell :c-border-risky] content)))))

(defn- no-alerts-row [{:keys [img-dir]}]
  (let [no-alerts-icon (styling/load-resource img-dir "Empty-alerts.png")
        no-alerts-text "Currently, there are no active risks to show."]
    (ph/row
      (ph/cell
        [:tta-na-cell]
        (ph/table
          [] [1 9]
          (ph/row
            (ph/cell [:tta-na-image-cell] (ph/image [:tta-na-image] no-alerts-icon))
            (ph/cell [:tta-na-text-cell] (ph/paragraph [:tta-na-text-paragraph] no-alerts-text))))))))

(defn- get-alert-parts [top-3-alerts options]
  (let [num-alerts (count top-3-alerts)]
    (case num-alerts
      0 {:column-widths nil
         :contents      (no-alerts-row options)}
      1 {:column-widths nil
         :contents      (->> top-3-alerts
                             (first)
                             (alert-contents options [16 1])
                             (alert-content-layout-single)
                             (alert-red-cell)
                             (ph/row))}
      2 {:column-widths [30 1 30]
         :contents (->> top-3-alerts
                        (map #(alert-contents options [3 14] %))
                        (map alert-content-layout-multiple)
                        (map alert-red-cell)
                        (interpose " ")
                        (apply ph/row))}
      3 {:column-widths [20 1 20 1 20]
         :contents (->> top-3-alerts
                        (map #(alert-contents options [5 12] %))
                        (map alert-content-layout-multiple)
                        (map alert-red-cell)
                        (interpose " ")
                        (apply ph/row))})))


(defn pdf-element [{:keys [early-warnings]} {:keys [img-dir dashboard-url] :as options}]
  (let [top-3-alerts (get-top-n-alerts early-warnings 3)
        {:keys [column-widths contents]} (get-alert-parts top-3-alerts options)]
    (ph/table [:c-card-content-top-table] nil
              (ph/row (ph/cell [] (ph/paragraph [:c-miniheader] "ACTIVE RISKS") [:spacer 1]))
              (ph/row (ph/cell [] (ph/table [] column-widths contents)))
              (ph/row (ph/cell [] [:spacer 1] (ph/line [:c-line]) [:spacer 1]))
              (ph/row (ph/cell [:tta-description-cell] (ph/paragraph [:c-subheader] "To see more details, please access your dashboard.")))
              (ph/row (ph/cell [] (ph/table [] [4 3] (pdf-elements/link-row dashboard-url "Go to Dashboard" img-dir)))))))