(ns codescene.features.recommendations.event-log.atom-log
  (:require
   [codescene.features.recommendations.definitions.recommendations :as recommendations]
   [codescene.features.recommendations.event-log.events :as events]
   [codescene.features.recommendations.event-log.queries :as q]
   [medley.core :as m])
  (:import
   (java.time Instant)))


(def event-stream #{:analysis :project :config :recommendation :user})

(defn- valid-event-to-append?
  [event]
  (and
    (map? event)
    (event-stream (:stream event))))

(defn- now []  (Instant/now))

(defn- filter-by-identifiers
  "If `recommendation-event` contains a list of `:identifiers`, then
  this function filters out any non-recommendation events whose
  `:identifier` field does not match one of the original items in the
  `:identifiers` list. The reason this only applies to
  non-recommendation events is that we want to keep the first event,
  which should be the recommendation event, and possibly the last
  event if it is the resolution event for that recommendation."
  [recommendation-event events]
  (let [identifiers-set (set (:identifiers recommendation-event))
        pred (fn [{:keys [stream identifier]}] (or (= :recommendation stream)
                                                   (identifiers-set identifier)))]
    (if (not-empty identifiers-set)
      (vec (filter pred events))
      events)))

(defn filtered-log-from-match
  "Builds a reducing function for `reduce-log-for-project`. 

  `stop-match` is a function that returns true when a log event should
  be the first item in the search space. In other words, since
  search (currently) starts at the present and works back in time,
  `stop-match` tells us when to stop searching.

  `limit-match` defines the end of the sequence. When getting the
  events for a recommendation that is resolved, then we want to stop
  including events after the resolution event. `limit-match`, when it
  matches, stops the search from matching any more recent events.

  The `keep-preds` are then applied to the remaining items (between
  the `stop-pred` match and the present) and determine which items
  should be returned, with an OR logic.

  The item matched by `stop-match` is always returned, even if none of
  the `keep-preds` match it."
  [stop-match limit-match keep-preds]
  (let [pred (if (not-empty keep-preds)
               (apply q/or-fn stop-match limit-match keep-preds)
               (constantly true))]
    (fn [events]
      (let [found-events (reverse
                           (into []
                                 (comp
                                   (m/take-upto stop-match)
                                   (filter pred))
                                 (reverse events)))
            initial-recommendation (first found-events)]
        ;; The first item must match the stop-predicate. This might
        ;; not be the case if the predicate events are shared between recommendations.
        (if (stop-match initial-recommendation)
          (->>
            ;; Restrict to events preceding the resolution event
            (m/take-upto limit-match found-events)
            (filter-by-identifiers initial-recommendation))
          [])))))


(defn stop-match-from [recommendation]
  (q/or-fn
   (q/map-as-predicate {:stream :recommendation :event (:id recommendation)})
   (q/map-as-predicate {:stream :project :event :project-creation})))

(defn limit-match-from [recommendation]
  (if-let [resolved (:resolved recommendation)]
    (q/map-as-predicate {:stream :recommendation :event resolved})
    (constantly false)))

(defn-  exclude-before-reset
  "Removes any items that occur before a :reset event (stream = :recommendation)."
  [events]
  ;; If we were worried about performance here, we would use reduce instead.
  (let [reset-event? (q/map-as-predicate {:stream :recommendation :event :reset})]
            (->> events
             (partition-by reset-event?)
             last
             (remove reset-event?))))

(defn- oldest-events-by-type
  [events]
  (->
   (reduce
    (fn [{:keys [keys-found] :as acc} {:keys [event] :as ev}]
      (cond-> acc
        (not  (contains? keys-found event))
        (-> (update :keys-found conj event)
            ;; this should preserve original order without a reverse
            (update :out #(conj % ev)))))
    {:keys-found #{} :out []}
    events)
   :out))

(defn- filter-by-project
  "Returns a transducer."
  [project-id]
  (filter #(= project-id (:project-id %))))

(defrecord AtomRecommendationEventLog [log]
  events/RecommendationEventLog
  (append-event [_this project-id event]
    (if (valid-event-to-append? event)
      (swap! log conj
             (assoc event
                    :project-id project-id
                    :created (now)))
      (throw (ex-info "Invalid event" {:bad-event event}))))

  (reduce-log-for-project [_this project-id reducing-fn]
    (->> @log
         exclude-before-reset
         (into [] (filter-by-project project-id))
         ((partial sort-by :created))
         reducing-fn))

  (reduce-log-for-project [_this project-id reducing-fn xfs]
    (let [xfilt (apply comp
                       (filter-by-project project-id)
                       xfs)]
      (->> @log
           exclude-before-reset
           (into [] xfilt)
           ((partial sort-by :created))
           reducing-fn)))
  (latest-related-events [this project-id user-id recommendation]
    (events/reduce-log-for-project
     this
     project-id
     (filtered-log-from-match
      (stop-match-from recommendation)
      (limit-match-from recommendation)
      (map q/map-as-predicate (recommendations/related-events-as-predicate-templates user-id recommendation)))))

  (resolved-project-recommendations [this project-id]
    (events/reduce-log-for-project
     this
     project-id
     oldest-events-by-type
     [(filter (q/map-as-predicate {:stream :recommendation :event (recommendations/resolved-project-recommendation-ids)}))]))

  (resolved-project-user-recommendations [this project-id user-id]
    (events/reduce-log-for-project
      this
      project-id
      oldest-events-by-type
      [(filter (q/map-as-predicate {:stream :recommendation
                                    :event (recommendations/resolved-user-recommendation-ids)
                                    :user-id user-id}))])))

;(events/resolved-project-recommendations (make-log) 5)
 

(defn make-log []
  (AtomRecommendationEventLog. (atom [])))

(defn make-test-log [events]
  (AtomRecommendationEventLog. (atom events)))

(defn- all-keys-true? [m]
  (every? second m))

(defn- prerequisites-met? [log project-id user-id recommendation]
  (let [required (-> recommendation :activate :prerequisites :required)
        required-resolved (map (fn [k] (:resolved (get recommendations/all k))) required)
        required-preds (map
                         (fn [k] (q/recommendation-as-prerequisite-predicate user-id (get recommendations/all k)))
                         required)]
    (->
     (events/reduce-log-for-project
       log
       project-id
       (fn handle-things [events]
         (transduce
           (filter (apply q/or-fn required-preds))
           (completing
             (fn reducing-fn [acc {:keys [event]}]
               (cond
                 (all-keys-true? acc)
                 (reduced acc)
                 (get acc event)
                 acc
                 :else
                 (assoc acc event true))))
           (into {} (map #(vector % false) required-resolved))
           events)))
     all-keys-true?)))






