(ns codescene.features.pm-data.youtrack.youtrack-provider
  "A PM Data Provider for fetching data from YouTrack."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider -uses-supported-work-types -uses-done-transitions -uses-done-transitions]]
            [codescene.features.pm-data.youtrack.youtrack-fetcher :as f]
            [codescene.features.pm-data.youtrack.cache :as cache]
            [codescene.features.pm-data.youtrack.youtrack-ticket-id-extractor :as x]
            [taoensso.timbre :as log]))

(defn- with-href [{:keys [api-url] :as _provider-def} issues]
  (let [->href #(format "%s/issue/%s" api-url %)]
    (map #(assoc % :href (->href (:id %)))
         issues)))

(defn- fetch-project [{:keys [api-client] :as provider-def} project since]
  (let [issues (->> (cache/fetch-issues since api-client project provider-def)
                    (with-href provider-def))]
    (log/infof "Import %d issues from Youtrack project %s" (count issues) project)
    issues))

(defn- get-pm-data
  [{:keys [api-client] :as provider-def} {:keys [since] :as _pm-data-context}]
  (let [{:keys [external-project-ids]} provider-def
        issues (->> external-project-ids
                    (mapcat #(fetch-project provider-def % since))
                    (into []))
        done-transitions (->> (f/fetch-resolved-states api-client)
                              (mapv :key))]
    {:issues       issues
     :provider-def (assoc provider-def :work-done-transition-names done-transitions)}))

(defn- get-configuration-data [{:keys [api-client] :as _provider-def}]
  (log/info "Fetching configuration data from Youtrack")
  {:work-types (f/fetch-issue-types api-client)
   :fields (f/fetch-period-fields api-client)
   ;; For Youtrack only in-progress-transitions are configured, and thus we only want 'unresolved' states here
   :transitions (f/fetch-unresolved-states api-client)
   :projects (f/fetch-projects api-client)})

(defn- validate-settings
  [{:keys [api-client] :as _provider-def}]
  (f/fetch-me api-client)
  nil)

(deftype YouTrackProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this _repo] (x/make-ticket-id-extractor provider-def))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def))
  (-validate-settings [_this] (validate-settings provider-def)))

(defmethod -create-pm-data-provider "youtrack"
  [provider-def pm-data-context]
  (->YouTrackProvider provider-def pm-data-context))

(defmethod -uses-supported-work-types "youtrack"
  [_provider-def]
  false)

(defmethod -uses-done-transitions "youtrack"
  [_provider-def]
  false)

(comment
  (def api-url "https://codescene.youtrack.cloud")
  (def api-client (codescene.features.client.api/->ExtraProperties {:api-url api-url} (System/getenv "YOUTRACK_TOKEN")))
  (get-configuration-data {:api-client api-client})
  (get-pm-data {:api-client api-client
                :api-url api-url
                :external-project-ids ["DEMO"]
                :work-in-progress-transition-names ["In Progress"]}
               (evolutionary-metrics.trends.dates/string->date "2023-05-25"))
  (validate-settings {:api-client api-client})
  )
