(ns codescene.features.pm-data.youtrack.youtrack-fetcher
  (:require [taoensso.timbre :as log]
            [clj-time.coerce :as tc]
            [clojure.string :as str]
            [codescene.features.pm-data.youtrack.youtrack-api :as api]
            [codescene.pm-data.provider-common :as common]
            [evolutionary-metrics.trends.dates :as dates]
            [medley.core :as m]))

(defn- millis->string
  [millis]
  (dates/date-time->string (tc/from-long millis)))

(defn- ->name-p [value] (fn [{:keys [name]}] (= name value)))
(defn- ->type-p [value] (fn [{:keys [$type]}] (= $type value)))
(defn- ->direction-p [value] (fn [{:keys [direction]}] (= direction value)))
(defn- ->link-type-p [value] (fn [{{:keys [name]} :linkType}] (= name value)))
(defn- ->field-type-p [value] (fn [{{:keys [id]} :fieldType}] (= id value)))

(defn- issue->response
  [{:keys [cost-field] :as _provider-def} {:keys [idReadable created updated resolved customFields summary] :as _issue}]
  (let [;; There should really be just one match for each of these custom fields
        ;; - though youtrack apparently supports multiple state fields
        ;; (see https://www.jetbrains.com/help/youtrack/devportal/api-entity-CustomField.html)
        cost (->> customFields (m/find-first (->name-p cost-field)) :value :minutes)
        state (->> customFields (m/find-first (->type-p "StateIssueCustomField")) :value :name)
        type (->> customFields (m/find-first (->name-p "Type")) :value :name)]
    {:id          (str/upper-case idReadable)
     :task-name   summary
     :created     (millis->string created)
     :closed      (when resolved (millis->string resolved))
     :updated     (millis->string updated)
     :status      state
     :cost        (or cost 0)
     :work-types  [type]}))

(defn- activity->state-transitions
  [{:keys [id added target timestamp] :as _activity}]
  (when (coll? added)
    ;; The 'added' property is an array for state transitions 
    ;; (would be suprising to see more than one element there though...)
    (keep (fn [{:keys [name $type]}]
            (when (= "StateBundleElement" $type)
              {:id        id
               :issue     (str/upper-case (:idReadable target))
               :state     name
               :timestamp (millis->string timestamp)}))
          added)))

(defn- issue->links
  [{:keys [idReadable updated links] :as _issue}]
  (let [;; Parent links are apparently accessible as INWARD links of type Subtask
        ;; (see https://www.jetbrains.com/help/youtrack/devportal/api-entity-IssueLink.html)
        parents (->> links 
                     (filter (->direction-p "INWARD"))
                     (filter (->link-type-p "Subtask")) 
                     (mapcat :issues)
                     (map :idReadable))]
    (for [parent parents]
      {:id      (str/upper-case idReadable)
       :parent  (str/upper-case parent)
       :updated (millis->string updated)})))

(defn- project->response
  [{:keys [shortName name] :as _project}]
  {:key shortName
   :name name})

(defn- bundle->values
  [{:keys [values] :as _state-bundle}]
  (map (comp common/->name-and-key :name) values))

(defn fetch-projects
  [api-client]
  (log/infof "Fetch projects from YouTrack")
  (->> (api/fetch-projects api-client)
       (mapv project->response)))

(defn- fetch-states
  [api-client pred]
  (log/infof "Fetch states from YouTrack")
  (->> (api/fetch-state-bundles api-client)
       (mapcat :values)
       (filter pred)
       (map (comp common/->name-and-key :name))
       distinct
       vec))

(defn fetch-resolved-states
  [api-client]
  (fetch-states api-client :isResolved))

(defn fetch-unresolved-states
  [api-client]
  (fetch-states api-client (complement :isResolved)))

(defn fetch-period-fields
  "Returns custom fields of the period type for use when cost-unit is set to minutes 
   Maybe there are point type fields too, meaning we could enable the points cost type
   - but for now this is good enough."
  [api-client]
  (log/info "Fetch period fields")
  (->> (api/fetch-custom-fields api-client)
       (filter (->field-type-p "period"))
       (mapv (comp common/->name-and-key :name))))

(defn fetch-issue-types
  "Fetches issue types.
   The type is a custom field, and here we get the possible values for it by
   - finding all instances of the custom field, and then
   - retrieving all distinct values in the bundles for each instance"
  [api-client]
  (log/info "Fetch issue types")
  (->> (api/fetch-custom-fields api-client)
       (filter (->name-p "Type"))
       (mapcat :instances)
       (map :bundle)
       (mapcat bundle->values)
       distinct
       vec))

(defn fetch-issues
  [since api-client project provider-def]
  (let [search-options {:project project
                        :since since
                        :fields "summary,created,updated,resolved,idReadable,customFields(id,name,value(name,minutes))"}]
    (log/infof "Fetch issues from YouTrack since %s" (or since "-"))
    (->> (api/search-issues api-client search-options)
         (mapv (partial issue->response provider-def)))))

(defn fetch-transitions
  [since api-client project]
  (let [search-options {:project project :since since}]
    (log/infof "Fetch issues from YouTrack since %s" (or since "-"))
    (->> (api/search-activities api-client search-options)
         (mapcat activity->state-transitions)
         vec)))

(defn fetch-links
  "Note that we make two separate searches for issues (if using the map-to-parent option).
   We could have written the code to return the links in the fetch-issues call, 
   but then this provider would have had to be structured different from the other providers (in the provider/extractor namespaces).
   And since the separate calls for activities when fetching issues is a much bigger problem anyway, let's not bother with that."
  [since api-client project]
  (let [search-options {:project project
                        :since since
                        :fields "updated,idReadable,links(direction,linkType(name),issues(idReadable))"}]
    (log/infof "Fetch links from YouTrack since %s" (or since "-"))
    (->> (api/search-issues api-client search-options)
         (mapcat issue->links)
         vec)))
 
(def fetch-me api/fetch-me)

(comment
  (def api-client (codescene.features.client.api/->ExtraProperties {:api-url "https://codescene.youtrack.cloud"} (System/getenv "YOUTRACK_TOKEN")))
  (def project "DEMO")
  (def since (evolutionary-metrics.trends.dates/string->date "2023-05-20"))
  (fetch-projects api-client)
  (fetch-resolved-states api-client)
  (fetch-unresolved-states api-client)
  (fetch-period-fields api-client)
  (fetch-issue-types api-client)
  (fetch-links since api-client project)
  (fetch-transitions since api-client project)
  (->> (fetch-issues since api-client project {})
       (filter #(= "DEMO-12" (:id %)))
       )
  (tc/from-long 1684918606399)
  )
