(ns codescene.features.pm-data.trello.trello-ticket-id-extractor
  "This namespace contains fns handling ticket id extraction from commit messages."
  (:require [clojure.string :as str]
            [codescene.features.pm-data.extractor-common :as ec]))

(def ticket-id-pattern #"([a-zA-Z][\w-]+)-(\d+)") ;; match ids like ABC-123 (or abc-123)

(defn scoped-ticket-id [{:keys [scope number] :as _ticket-id}]
  (format "%s-%s" (str/upper-case scope) number))

(defn extract-ticket-ids [msg]
  (->> (re-seq ticket-id-pattern msg)
       (map (fn [[_ scope number]] {:scope scope :number number}))))

(defn make-ticket-id-extractor
  "Creates an extractor that matches ticket ids for the configured Trello boards."
  [{:keys [external-project-ids] :as provider-def}]
  (let [aliases (ec/aliases-from provider-def)
        ->pr-ticket-ids (ec/make-pr-ticket-ids-lookup provider-def extract-ticket-ids)]
    (fn [rev _date-time msg]
      (when (seq external-project-ids)
        (->> (concat (extract-ticket-ids msg)
                     (->pr-ticket-ids (subs rev 0 7)))
             (filter (ec/make-ticket-filter external-project-ids aliases))
             (map #(ec/with-resolved-aliases aliases %))
             (map scoped-ticket-id))))))


(comment
  (def context {:repo-urls ["git@github.com:empear-analytics/codescene.git"]})
  (def api-client (codescene.features.pm-data.trello.trello-api/trello-auth
                   (System/getenv "TRELLO_API_KEY")
                   (System/getenv "TRELLO_API_TOKEN")))
  (def x (let [api-client (codescene.features.pm-data.trello.trello-api/trello-auth
                           (System/getenv "TRELLO_API_KEY")
                           (System/getenv "TRELLO_API_TOKEN"))]
           (make-ticket-id-extractor {:api-client api-client
                                      :external-project-ids ["CodeScene - Test Automation"]
                                      :project-aliases [["CSTA" "CodeScene - Test Automation"]]})))
  (x "12345678" nil "CSTa-1")
  (def board-name "CodeScene - Test Automation")
  (def board-id "6045ea5748782b336a1feafb")
)