(ns codescene.features.pm-data.pr-data-provider
  "Implementation protocol for pr data providers."
  (:require [clojure.spec.alpha :as s]
            [codescene.specs :as specs]))

(defprotocol PrDataProvider
  (-validate-settings [this])
  (-get-pull-requests [this since]))

(defmulti -create (fn [provider-def _pr-data-context] (:type provider-def)))

(s/def ::id string?)
(s/def ::title string?)
(s/def ::branch string?)
(s/def ::text (s/nilable string?))
(s/def ::commits (s/coll-of string?))
(s/def ::updated ::specs/datetime)
(s/def ::merge-commit (s/nilable string?))
(s/def ::pull-request (s/keys :req-un [::id
                                       ::text
                                       ::commits]
                              :opt-un  [::title
                                        ::branch
                                        ::merge-commit
                                        ::updated]))
(s/def ::pull-requests (s/coll-of ::pull-request))
(s/def ::type string?)
(s/def ::provider-def (s/keys :req-un [::type]))
(s/def ::repo-urls (s/coll-of string?))
(s/def ::pr-data-context (s/keys :req-un [::repo-urls]))
(s/def ::pr-data-provider #(satisfies? PrDataProvider %))

(s/fdef validate-settings
  :args (s/cat :provider ::pr-data-provider))

(defn validate-settings
  [provider]
  (-validate-settings provider))

(s/fdef get-pull-requests
  :args (s/cat :provider ::pr-data-provider
               :since (s/nilable ::specs/datetime))
  :ret ::pull-requests)

(defn get-pull-requests
  [provider since]
  (-get-pull-requests provider since))

(s/fdef create
  :args (s/cat :provider-def ::provider-def
               :pr-data-context ::pr-data-context)
  :ret ::pr-data-provider)

(defn create
  "Create a provider according to the def"
  [provider-def pr-data-context]
  (-create provider-def pr-data-context))
