(ns codescene.features.pm-data.clickup.fetcher
  "Uses the Jira API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [taoensso.timbre :as log]
            [codescene.cache.core :as cache]
            [codescene.features.pm-data.clickup.api :as api]
            [codescene.pm-data.provider-common :as common]
            [clj-time.coerce :as coerce]
            [clojure.string :as str]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- fetch-spaces
  [api-client {:keys [id name] :as _team}]
  (log/info "Fetch spaces in" name)
  (api/fetch-spaces api-client id))

(defn- fetch-folders
  [api-client {:keys [id name] :as _space}]
  (log/info "Fetch folders in" name)
  (api/fetch-folders api-client id))

(defn- fetch-folderless-lists
  [api-client {:keys [id name] :as _space}]
  (log/info "Fetch folderless lists in" name)
  (api/fetch-folderless-lists api-client id))

(defn- fetch-custom-task-types
  [api-client {:keys [id name] :as _team}]
  (log/info "Fetch custom task types in" name)
  (api/fetch-custom-task-types api-client id))

(defn- fetch-list
  [api-client {:keys [id name] :as _list}]
  (log/info "Fetch list" name)
  (api/fetch-list api-client id))

(defn- status->key-and-name
  [{:keys [status] :as _status}]
  (common/->name-and-key (str/upper-case status)))

(defn- statuses-in
  [{:keys [statuses] :as _item}]
  (->> (map status->key-and-name statuses)
       (into [])))

(defn- fetch-list-statuses
  [api-client {:keys [override_statuses] :as list}]
  (if override_statuses
    (statuses-in (fetch-list api-client list))
    []))

(defn- folder-statuses
  [{:keys [lists] :as folder}]
  (concat
   (statuses-in folder)
   (for [list lists
         s (statuses-in list)]
     s)))

(defn- fetch-space-statuses
  [api-client space]
  (->> (concat
        (statuses-in space)
        (for [list (fetch-folderless-lists api-client space)
              s (fetch-list-statuses api-client list)]
          s)
        (for [folder (fetch-folders api-client space)
              s (folder-statuses folder)]
          s))
       distinct
       (into [])))

(defn- make-task-type-name-lookup
  [api-client team]
  (let [task-types (fetch-custom-task-types api-client team)]
    (->> task-types
         (map (juxt :id :name))
         (into {0 "Task"})))) ;; For some reason, the default type is not returned by the API

(defn- safe-from-long-str
  [s]
  (some-> s Long/parseLong coerce/from-long dates/date-time->string))

(defn- time-in-status->transitions
  [{:keys [status_history] :as _time-in-status}]
  (->> status_history
       (map (fn [{:keys [status total_time]}]
              [(str/upper-case status) (-> total_time :since safe-from-long-str)]))
       reverse ; The dates are from new to old -> reverse
       (into [])))

(defn- task->response
  [tasks-time-in-status
   task-type-id->name
   {:keys [id custom_id custom_item_id parent date_created date_updated date_closed status url] :as task}]
  (let [time-in-status (tasks-time-in-status id)]
    {:id          (or custom_id id)
     :internal-id id
     :parent      parent
     :task-name   (:name task)
     :href        url
     :created     (safe-from-long-str date_created)
     :closed      (safe-from-long-str date_closed)
     :updated     (safe-from-long-str date_updated)
     :status      (-> status :status str/upper-case)
     :cost        0
     :transitions (time-in-status->transitions time-in-status)
     :work-types  [(get task-type-id->name custom_item_id "unknown")]}))

(defn- fetch-tasks-for-team*
  [since api-client {:keys [id name] :as team} space-ids]
  (let [search-options {:spaces space-ids
                        :since since}]
    (log/infof "Fetch tasks in %s from ClickUp since %s" name (or since "-"))
    (let [tasks (api/fetch-filtered-team-tasks api-client id search-options)
          tasks-time-in-status (api/fetch-bulk-tasks-time-in-status api-client (map :id tasks))
          task-type-id->name (make-task-type-name-lookup api-client team)]
      (map (partial task->response tasks-time-in-status task-type-id->name) tasks))))

(defn- valid-space-ids
  [api-client team space-ids]
  (let [spaces (fetch-spaces api-client team)]
    (filter (->> spaces (map :id) set) space-ids)))

(defn- fetch-tasks-for-team
  [since api-client team space-ids]
  (let [valid-space-ids (not-empty (valid-space-ids api-client team space-ids))]
    (if (seq valid-space-ids)
      (fetch-tasks-for-team* since api-client team valid-space-ids)
      [])))

(defn fetch-teams
  [api-client]
  (log/info "Fetch teams")
  (api/fetch-teams api-client))

(defn fetch-all-spaces
  [api-client]
  (->> (for [{team-name :name :as team} (fetch-teams api-client)
             {:keys [id name] :as space} (fetch-spaces api-client team)]
         {:key id
          :name (format "%s/%s" team-name name)
          ;; A bit of a hack this - include space in response so it can be used in subsequent call
          :space space})
       (into [])))

(defn fetch-all-task-types
  [api-client]
  (log/info "Fetch task types")
  (let [task-types (->> (fetch-teams api-client)
                        (mapcat #(fetch-custom-task-types api-client %)))]
    (->> task-types
         (map :name) ;; all providers use name as key too, so do the same with this one
         (into ["Task"]) ;; For some reason, the default type is not returned by the API
         distinct
         (mapv common/->name-and-key))))

(defn fetch-all-space-statuses
  [api-client spaces]
  (->> spaces
       (mapcat #(fetch-space-statuses api-client (:space %)))
       distinct
       (into [])))

(defn fetch-tasks
  [since api-client space-ids]
  (let [teams (fetch-teams api-client)]
    (->> teams
         (mapcat #(fetch-tasks-for-team since api-client % space-ids))
         (into []))))
    
(cache/memo-scoped #'fetch-teams)
(cache/memo-scoped #'fetch-all-spaces)
(cache/memo-scoped #'fetch-all-task-types)
(cache/memo-scoped #'fetch-all-space-statuses)

(comment
  (def api-client (api/clickup-auth (System/getenv "CLICKUP_TOKEN")))
  (def since (evolutionary-metrics.trends.dates/string->date "2025-02-12"))
  (fetch-all-task-types api-client)
  (fetch-all-spaces api-client)
  
  (->> (fetch-all-spaces api-client)
       (fetch-all-space-statuses api-client))
  
  (def tasks (fetch-tasks since api-client ["90152376032"] #_(map :key lists)))
  (count tasks)
  (take 10 tasks))
  
