(ns codescene.features.pm-data.azure.azure-ticket-id-extractor
  "This namespace contains fns handling Azure Work Item id extraction and generic ticket id extraction 
   from Azure PRs."
  (:require [clojure.string :as str]
            [codescene.features.pm-data.azure.azure-cache :as ac]
            [codescene.features.pm-data.extractor-common :as ec]
            [codescene.pm-data.provider-common :as c]
            [codescene.url.url-utils :as url]))

;; match issue nbrs with optional scope, ie '#123' as well as 'AB#123' 
(def ticket-id-pattern #"([\w-]+)?#(\d+)")

(defn scoped-ticket-id [{:keys [scope number] :as _ticket-id}]
  (format "%s#%s" (str/lower-case scope) number))

(defn ->ticket-id-scoper [{:keys [owner] :as _project-parts}]
  ;; Issues are per project, but ids are unique across owner (org) 
  ;; -> create a scoper that use just the owner part when scoping issue ids
  (fn [number]
    (scoped-ticket-id {:scope owner :number number})))

(defn- fetch-links [{:keys [api-client] :as provider-def} project-parts]
  (let [->scoped-ticket-id (->ticket-id-scoper project-parts)]
    (->> (ac/fetch-links nil api-client project-parts provider-def)
         (map #(-> %
                   (update :id ->scoped-ticket-id)
                   (update :parent ->scoped-ticket-id))))))

(defn- make-parent-lookup [provider-def project-parts-coll]
  (if (:map-subtasks-to-parent-issues provider-def)
    (->> project-parts-coll
         (mapcat (partial fetch-links provider-def))
         c/->parent-lookup)
    {}))

(defn- with-resolved-parents [->parent id]
  (get ->parent id id))
  
(defn- ->scoped-ticket-id
  "Uses the capture groups from the match to decide on how to construct a scoped ticket id"
  [owner {repo-owner :owner :as _repo-parts} [_match scope number]]
  (cond
    ;; No scope allowed for azure
    (and (= owner repo-owner) (not scope)) {:scope owner :number number}
    ;; Must be ab-scope if non-azure
    (and (not= owner repo-owner) (= "AB" scope)) {:scope owner :number number}
    :else nil))

(defn extract-ticket-ids [owner repo-parts msg]
  (if msg
    (->> msg
         (re-seq ticket-id-pattern)
         (keep #(->scoped-ticket-id owner repo-parts %)))
    []))

(defn make-pr-ticket-ids-lookup [provider-def owner repo-parts]
  (ec/make-pr-ticket-ids-lookup provider-def (partial extract-ticket-ids owner repo-parts)))

(defn make-ticket-id-extractor [provider-def repo-url project-parts-coll _pm-data-context]
  (let [owner (some :owner project-parts-coll)
        repo-parts (url/azure-url->parts* repo-url)
        ->parent (make-parent-lookup provider-def project-parts-coll)
        ->pr-ticket-ids (make-pr-ticket-ids-lookup provider-def owner repo-parts)]
    (fn [rev _date-time msg]
      (->> (concat (extract-ticket-ids owner repo-parts msg)
                   (->pr-ticket-ids (subs rev 0 7))) ;; TODO: Use entire hash
           (map scoped-ticket-id)
           (map #(with-resolved-parents ->parent %))))))

(comment
  (def repo-url "git@ssh.dev.azure.com:v3/empear/SmartHotel360/PublicWeb")
  (def repo-url "git@ssh.dev.azure.com:v3/empear/Project%20With%20Spaces/Repo%20With%20Spaces")
  (def api-client ["" (System/getenv "AZURE_TOKEN")])
  (def pr-data-provider (codescene.features.pm-data.azure.pr-data-provider/->AzureProvider
                         {:api-client api-client}
                         {:repo-urls [repo-url]}))
  (def provider-def {:map-commits-to-pull-request-issues "on"
                     :map-subtasks-to-parent-issues "on"
                     :api-client api-client
                     :pr-data-provider pr-data-provider})
  (def repo-parts (url/azure-url->parts repo-url))
  (def project-parts-coll [(url/azure-url->parts repo-url)])
  (make-parent-lookup provider-def project-parts-coll)
  (make-pr-ticket-ids-lookup provider-def "empear" repo-parts)
  (def ex (make-ticket-id-extractor provider-def repo-url project-parts-coll {}))
  (ex "1bfa7ec" nil "AB#1")
  )
