(ns codescene.features.plugins.system-map-extension
  "Wraps an external SystemMapExtension in an implementation of 
  the CustomMetricsProvider protocol defined in codescene-analysis."
  (:require [codescene.features.plugins.configuration :as c]
            [clj-codescene-plugin.metric :as metric]
            [codescene.custom-metrics.custom-metrics-provider :refer [CustomMetricsProvider]]
            [clj-codescene-plugin.system-map :as system-map]
            [clojure.string :as str]
            [taoensso.timbre :as log]))

(defn- ->hotspot-data [{:keys [module] :as _hotspot}]
  (when-let [i (str/index-of module "/")]
    {:module module
     :repo (subs module 0 i)
     :path (subs module (inc i))}))

(defn- ->Context
  "Provides a Context instance for the external SystemMapExtension"
  [{:keys [repo-paths settings encryptor] :as _context}]
  (system-map/->Context {:repositories repo-paths
                         :settings (c/decrypt-values encryptor settings)}))

(defn ->File 
  "Provides a File instance for the external SystemMapExtension"
  [{:keys [path] :as _hotspot-data}]
  (system-map/->File {:path path}))
    
(defn ->Repository
  "Provides a Repository instance for the external SystemMapExtension"
  [{:keys [repo] :as _hotspot-data}]
  (system-map/->Repository {:name repo
                            :path repo
                            :origin-url repo}))

(defn- score-hotspot [analyzer metrics repo {:keys [module] :as hotspot}]
  (let [scores (.score analyzer repo (->File hotspot))
        interpret-score  (fn [[metric score]] [(:id metric) (metric/value metric score)])]
    {:module module
     ;; Create a seq of [metric value] pairs and then interpret the values
     :scores (->> (map vector metrics scores)
                  (map interpret-score)
                  (into {}))}))
   
(defn- score-repo [plugin-id analyzer metrics hotspots]
  (let [repo (-> hotspots first ->Repository)
        scores (map #(score-hotspot analyzer metrics repo %) hotspots)
        any-file-score? (fn [file-scores] (some some? file-scores))
        any-score? (->> scores (map :scores) (some any-file-score?))]
    (when-not any-score?
      (log/warnf "No scores found by plugin %s in repo %s" plugin-id (-> hotspots first :repo)))
    scores))

(defn- score-hotspots [plugin-id analyzer metrics hotspots]
  (try
    ;; TODO: This check should really be in codescene.custom-metrics.core before calling score-hotspots
    (if (seq metrics) 
      (let [partitioned-by-repo (->> hotspots (map ->hotspot-data) (partition-by :repo))]
        (-> (mapcat #(score-repo plugin-id analyzer metrics %) partitioned-by-repo)
            doall))
      [])
    (catch Exception e
      (let [msg (format "Failed to score-hotspots in plugin %s" plugin-id)]
        (log/warnf e msg)
        []))))

(defn- metrics [plugin-id analyzer]
  (let [prepend-plugin-id #(str plugin-id "-" %)]
    (try
      (->> (.getMetrics analyzer)
           (map #(metric/Metric->map %))
           (map #(update % :id prepend-plugin-id))
           doall)
      (catch Exception e
        (let [msg (format "Failed to get metrics in plugin %s" plugin-id)]
          (log/warnf e msg)
          [])))))

(defn ->CustomMetricsProvider [plugin-id extension context]
  (try
    (let [analyzer (.getAnalyzer extension (->Context context))
          metrics (metrics plugin-id analyzer)]
      (reify CustomMetricsProvider
        (-id [_this] plugin-id)
        (-metrics [_this] metrics)
        (-score-hotspots [_this _code-properties-fn hotspots] (score-hotspots plugin-id analyzer metrics hotspots))
        (-validation-results [_this] [])))
    (catch Exception e
      (let [msg (format "Failed to create extension in plugin %s" plugin-id)]
        (log/warnf e msg)))))

(comment
  (def hotspots [{:module "analysis-target/a/c-example.c"
                  :revisions "2"
                  :code "5"
                  :id "analysis-target/c-example.c"}])
  (def context {:repo-paths []
                :global-settings []
                :project-settings []})
  (def metric (reify com.codescene.plugin.systemmap.Metric
                (getType [_] com.codescene.plugin.systemmap.MetricType/INT)
                (getId [_] "int-metric")
                (getName [_] "Int Metric")
                (getUnit [_] "%")
                (getMinValue [_] (com.codescene.plugin.systemmap.OptionalValue/ofInt 0))
                (getMaxValue [_] (com.codescene.plugin.systemmap.OptionalValue/ofInt 100))
                (isSummable [_] true)
                (isPositive [_] true)
                (getColor [_] (java.util.OptionalInt/empty))))
  (def ext (reify com.codescene.plugin.SystemMapExtensionPoint
             (getAnalyzer [_ _] 
               (reify com.codescene.plugin.systemmap.Analyzer
                 (getMetrics [_] 
                   [metric])
                 (score [_ _ _] [(com.codescene.plugin.systemmap.OptionalValue/ofInt 42)])))))
  (def e (->CustomMetricsProvider "dummy-plugin" ext context))
  (codescene.custom-metrics.custom-metrics-provider/score-hotspots e (constantly nil) hotspots))
