(ns codescene.features.plugins.plugin-manager
  "Wraps the Java interop related to the pf4j plugin manager.
   We create a custom plugin manager that
   - loads jar plugins only
   - find plugin descriptions in manifests only
   - loads only plugins that require the exact codescene-plugin 
     version we depend on"
  (:require [codescene.features.plugins.meta-inf :as meta-inf]
            [clojure.java.io :as io]
            [taoensso.timbre :as log]
            [semver.core :as semver])
  (:import (com.codescene.plugin SystemMapExtensionPoint)
           (org.pf4j DefaultPluginManager JarPluginLoader ManifestPluginDescriptorFinder
                     DefaultVersionManager)))

(def system-version (meta-inf/file-contents SystemMapExtensionPoint "codescene-plugin.version"))

(defn- plugin-root-paths [env]
  (if-let [dir (:plugins-dir env)]
    [(.toPath (io/file dir))]
    []))

(defn ->VersionManager []
  (proxy [DefaultVersionManager] []
    (checkVersionConstraint [version constraint]
      (if (and (string? version) (string? constraint)
               (semver/valid? version) (semver/valid? constraint))
        (let [{major-version :major minor-version :minor} (semver/parse version)
            {major-constraint :major minor-constraint :minor} (semver/parse constraint)]
          (and (= major-version major-constraint) (>= minor-version minor-constraint)))
        false))))

(defn safe-set-system-version [plugin-manager]
  (when system-version
    (.setSystemVersion plugin-manager system-version)))

(defn ->PluginManager [env]
  (doto
   (proxy [DefaultPluginManager] [(plugin-root-paths env)]
     (createPluginLoader []
       (JarPluginLoader. this))
     (createPluginDescriptorFinder [] (ManifestPluginDescriptorFinder.))
     (createVersionManager [] (->VersionManager)))
    ;;We will only load plugins with the exact same version
    safe-set-system-version
    (.setExactVersionAllowed true)
    ;; The create* methods are called from the base class constructor,
    ;; something that proxy doesn't handle...
    ;; Thus, we need an extra initialize call to provide custom create* methods!
    .initialize))

(defn init [env]
  (let [plugin-manager (->PluginManager env)]
    (doto plugin-manager .loadPlugins .startPlugins)))

(defn shutdown [plugin-manager]
    (doto plugin-manager .stopPlugins .unloadPlugins))

(defn- Plugin->map [plugin]
  (let [desc (.getDescriptor plugin)]
    {:id (.getPluginId desc)
     :version (.getVersion desc)
     :description (.getPluginDescription desc)
     :class (.getPluginClass desc)
     :license (.getLicense desc)
     :provider (.getProvider desc)
     :dependencies (.getDependencies desc)
     :requires (.getRequires desc)
     ;; States are described in the docs and are typically STARTED or DISABLED
     ;; See https://pf4j.org/doc/plugin-lifecycle.html
     :state (.getPluginState plugin)}))

(defn get-plugins
  [plugin-manager]
  (->> (.getPlugins plugin-manager)
       (map Plugin->map)))

(defn get-extensions
  ([plugin-manager extension-point]
   (try
     (.getExtensions plugin-manager extension-point)
     (catch Exception e
       (log/warnf e "Failed to instantiate %s extensions. Error: " extension-point))))
  ([plugin-manager extension-point plugin-id]
   (try
     (.getExtensions plugin-manager extension-point plugin-id)
     (catch Exception e
       (log/warnf e "Failed to instantiate %s extensions in plugin %s. Error: " extension-point plugin-id)))))

(comment
  (:plugins-dir env)
  (def pm (init {:plugins-dir "./plugins"}))
  (shutdown pm)
  (get-extensions pm SystemMapExtensionPoint)
  (get-plugins pm)
  (->> pm clojure.reflect/reflect :members (map :name)))
