(ns codescene.features.config.cov-config
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [codescene.features.util.maps :as maps]
            [codescene.features.api.spec.code-coverage :as cc-spec]
            [codescene.features.config.properties :refer [enum bool sval property ival] :as p]
            [codescene.features.config.sets :as sets]
            [spec-tools.data-spec :as ds]))

(s/def ::qg-preset #{"global" "minimal" "medium" "all" "custom" "custom-file"})

(def properties
  {:github []
   :azure []
   :bitbucket []
   :gitlab []})

(def qg-properties [(property :qg-preset :qg-preset (enum ::qg-preset) "global")
                    (property :overall-coverage? :qg-overall-coverage (bool) false)
                    (property :new-and-changed-code? :qg-new-and-changed-code (bool) true)
                    (property :no-decline-in-coverage? :qg-no-decline-in-coverage (bool) false)
                    (property :new-file-coverage? :qg-new-file-coverage (bool) false)
                    (property :hotspot-must-improve? :qg-hotspot-must-improve (bool) false)
                    (property :modified-must-improve? :qg-modified-must-improve (bool) false)
                    (property :hotspot-coverage-target? :qg-hotspot-coverage-target (bool) false)])

(def settings-properties
  (into [(property :ignore-other-branches :cov-ignore-other-branches (bool) false)
         (property :coverage-metric :cov-coverage-metric (enum ::cc-spec/metric) "line-coverage")
         (property :overall-threshold :cov-overall-threshold (ival) 80)
         (property :changed-threshold :cov-changed-threshold (ival) 95)]
        qg-properties))

(def qg-properties-keys (mapv :id qg-properties))
(def qg-properties-keys-wo-preset (next qg-properties-keys))

(def general-properties [(property :enabled? :pr-cov-enabled (bool) false)])

(def cloud-all-providers-properties (concat general-properties
                                            settings-properties))

(defn profile [profile-name props]
  (let [all-false (zipmap qg-properties-keys-wo-preset (repeat false))
        all-true (zipmap qg-properties-keys-wo-preset (repeat true))]
    (merge all-false
           (get {"minimal" {:overall-coverage? true
                            :new-and-changed-code? true}
                 "medium" {:overall-coverage? true
                           :new-and-changed-code? true
                           :no-decline-in-coverage? true
                           :new-file-coverage? true}
                 "all" all-true
                 "custom" props
                 "custom-file" {}}
                profile-name))))

(defn merge-configuration
  [props global-config]
  (merge props
         (if (= "global" (:qg-preset props))
           (merge-configuration (:kv global-config) nil)
           (let [qg-preset (:qg-preset props "minimal")
                 custom-props (select-keys props qg-properties-keys-wo-preset)]
             (profile qg-preset custom-props)))))

(def cloud-properties (update-vals properties #(into % cloud-all-providers-properties)))

(defn- bitbucket-server-api-url-fix [url]
  (let [slash-trimmed (str/replace-first url #"/+$" "")]
    (if (or (= url "") (str/ends-with? slash-trimmed "/rest/api/1.0"))
      slash-trimmed (str slash-trimmed "/rest/api/1.0"))))

(def onprem-properties
  (-> (m/map-keys #(get {:github "github-app" :github-hook "github"} % (name %)) properties)
      (meta-merge
        {"github-app" [(property :secret :github-secret (sval))
                       (property :private-key :github-private-key (p/file-string true))
                       (property :app-id :github-app-id (sval))
                       (property :preset-id :github-preset-id (ival))
                       (property :api-url :github-api-url (sval) "")]
         "github" [(property :password :github-deprecated-token (sval true))
                   (property :api-url :github-deprecated-api-url (sval) "")
                   (property :host-url :github-deprecated-host-url (sval))]
         "azure" [(property :password :azure-token (sval true))
                  (property :api-url :azure-api-url (sval) "")
                  (property :preset-id :azure-preset-id (ival))
                  (property :host-url :azure-host-url (sval))]
         "bitbucket" [(property :password :bitbucket-password (sval true))
                      (property :api-url :bitbucket-api-url (sval) "")
                      (property :host-url :bitbucket-host-url (sval))
                      (property :preset-id :bitbucket-preset-id (ival))
                      (property :username :bitbucket-username (sval))]
         "bitbucket-server" [(property :build? :bitbucket-server-build (bool) true)
                             (property :report? :bitbucket-server-report (bool) true)
                             (property :annotations? :bitbucket-server-annotations (bool) true)
                             (property :password :bitbucket-server-token (sval true))
                             (assoc (property :api-url :bitbucket-server-api-url (sval) "")
                               :fix-fn bitbucket-server-api-url-fix)
                             (property :preset-id :bitbucket-server-preset-id (ival))
                             (property :host-url :bitbucket-server-host-url (sval))]
         "gerrit" [(property :password :gerrit-password (sval true))
                   (property :api-url :gerrit-api-url (sval) "")
                   (property :username :gerrit-username (sval))
                   (property :comments? :gerrit-comments (bool) false)
                   (property :origin-url :gerrit-origin-url (sval))
                   (property :host-url :gerrit-host-url (sval))
                   (property :preset-id :gerrit-preset-id (ival))
                   (property :user-hook? :gerrit-user-hook (bool) false)]
         "plugin-gerrit" {}
         "gitlab" [(property :password :gitlab-token (sval true))
                   (property :api-url :gitlab-api-url (sval) "")
                   (property :host-url :gitlab-host-url (sval))
                   (property :preset-id :gitlab-preset-id (ival))]})))

(defn flat-config [config] (-> config (merge (:config config)) (dissoc :config)))

(defn api-spec
  "API Spec for all configs"
  [provider-property-maps]
  (let [key-spec-coll (for [props (vals provider-property-maps)
                            {:keys [api-spec ui-param]} props]
                        [(ds/opt (maps/->db-name ui-param)) api-spec])]
    ;; the name doesn't matter, but it does show up in Swagger, so we pick a nice public one
    (ds/spec {:spec (into {} key-spec-coll) :name :pr-integration/config})))

;; Onprem pr-config has some keys at base then it has a :config key that leads to another
;; kv map
(defn ->api-representation
  "Convert webhook provider to params similar to UI"
  [all-types-map config]
  (let [config-set (sets/typed-config-set all-types-map (flat-config config) :live)]
    (:data (sets/to-ui-params config-set))))

(defn from-db-cloud
  [provider-id data]
  (p/from-db (get cloud-properties provider-id) data))

(defn from-db-onprem
  [provider-type data]
  (p/from-db (get onprem-properties provider-type) data))

(defn from-db [props data] (p/from-db props data))