(ns codescene.features.code-coverage.pr-check
  (:require [clojure.java.jdbc :as jdbc]
            [clojure.string :as str]
            [codescene.features.project.core :as project.core]
            [codescene.features.code-coverage.check.check-db :as db]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.util.template :as template]
            [selmer.parser :as selmer]
            [taoensso.timbre :as log]))

(def result-coord db/result-coord)
(def insert-pr-check db/insert-check)

(def gate-names
  {"overall_coverage" "Overall Coverage"
   "new_and_changed_code" "New & Changed Code Coverage"})


(defn awaiting-result-for-id!
  "Given an ID it will return the coverage result row, if it is awaiting
  results, and it will mark it as resolved, otherwise it will return nil"
  [db-spec id]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [result (db/get-by-id tx id)]
      (when (and result (not (:coverage-results result)))
        (log/info "Found code coverage result for %s" (result-coord result))
        (db/update-with-results tx (:project-id result) (assoc result :coverage-results {}))
        result))))

(defprotocol PrCheckProvider
  (-provider-id [this])
  (-post-result [this result metadata]))

(defn post-result [tx pr-check-provider project-id {:keys [coverage-results base-ref commit-sha repo-id] :as result}]
  (if-let [item (db/get-by-props tx project-id result)]
    (do (log/infof "Found check %s for project-id=%s, repo-id=%s, sha=%s, base-ref=%s"
                   (:id item) project-id repo-id commit-sha base-ref)
        (-post-result pr-check-provider
                      (merge (result-coord item)
                             {:coverage-results coverage-results})
                      (:metadata item)))
    (let [msg (format "Couldn't find check to update for project-id=%s, repo-id=%s, sha=%s, base-ref=%s"
                      project-id repo-id commit-sha base-ref)]
      (log/info msg)
      (throw (ex-info msg {:type :not-found})))))

(def timeout-gate
  {:pass false
   :measured-coverage nil
   :details {:pass-or-fail-reason "No valid coverage report found in the build pipeline"
             :action "Check your coverage configuration in the pipeline. CodeScene's CLI tools need to be given a valid code coverage report."}})

(defn post-timeout
  "Post result as a timeout if no result has been posted. "
  [db-spec check-id project-id->pr-check-provider gates]
  (if-let [{:keys [project-id metadata] :as result} (awaiting-result-for-id! db-spec check-id)]
    (let [result-per-gate (mapv #(merge % timeout-gate) gates)]
      (log/infof "PR Coverage posting timeout result for check %s on project %s" check-id project-id)
      (-post-result (project-id->pr-check-provider project-id)
                    (merge result
                           {:coverage-results {:all-gates-pass false
                                               :result-per-gate result-per-gate}})
                    metadata))
    (log/infof "PR Coverage check %s is already done" check-id)))

(defn iconset [provider-id]
  (let [url #(format "[![](https://codescene.io/imgs/svg/%s)](#)" %)]
    (case provider-id
      :github {:h2-pass "✅" :pass (url "pass1.svg") :fail (url "x1.svg") :h2-fail "❌"}
      (:gitlab :azure) {:h2-pass (url "pass.svg") :pass (url "pass1.svg") :fail (url "x1.svg") :h2-fail (url "x.svg")}
      {:h2-pass "✅" :pass "✅" :fail "❌" :h2-fail "❌"})))

(defn provider-options [provider-id]
  {:bottom-hr (case provider-id
                :azure "<hr>"
                :bitbucket "\n"
                "##      ")
   :icons (iconset provider-id)
   :collapsible (if (#{:bitbucket :gerrit} provider-id) false true)
   :extra-newline (if (#{:bitbucket :gitlab :azure} provider-id) "\n" "")})


(defn check-status [coverage-results]
  (let [enabled (filterv :enabled (:result-per-gate coverage-results))]
    (cond
      (empty? enabled) :disabled
      (every? :pass enabled) :success
      :else :fail)))

(defn render-action
  [action]
  (cond
    (nil? action) ""
    (string? action) (selmer/render action {})
    (map? action) (template/render-file "templates/coverage/action.md"
                                        {:action action})
    :else action))

(defn- render [{:keys [name threshold details measured-coverage]}]
  {:gate-name (gate-names name name)
   :desc (format "required = %d%%" threshold)
   :stat (if measured-coverage (str measured-coverage "%") "-")
   :reason (:pass-or-fail-reason details)
   :action (render-action (:action details))})

(defn result->renderable
  "Process result into a data structure that fits our rendering."
  [{:keys [result-per-gate] :as res}]
  (let [{fail false pass true} (->> result-per-gate
                                    (filterv :enabled)
                                    (group-by :pass))]
    {:qg-meta {:empty-results? false :status (check-status res)}
     :qg-summary {:fail (mapv render fail)
                  :pass (mapv render pass)}}))

(defn title [renderable provider-id]
  (if (= :fail (get-in renderable [:qg-meta :status]))
    (format "%s Code Coverage Gate Failed" (:h2-fail (iconset provider-id)))
    (format "%s Code Coverage Gate Passed" (:h2-pass (iconset provider-id)))))

(defn summary
  "Provider settings are bottom-hr collapsible ex"
  [renderable provider-id]
  (template/render-file "templates/coverage/summary.md"
                        (merge renderable (provider-options provider-id))))

(def timeout-title "PR Coverage Gates Timed Out")

(defn timeout-summary
  []
  "CodeScene PR Coverage check has timed out waiting for results. Check your integration that calculates the coverage.")

(defn itemized [renderable provider-id]
  (template/render-file "templates/coverage/itemized.md"
                        (merge renderable (provider-options provider-id))))

(defn create-message [renderable provider-id]
  (str (title renderable provider-id) "\n"
       (-> (provider-options provider-id) :bottom-hr) "\n"
       (summary renderable provider-id) "\n\n"
       (itemized renderable provider-id)))

(defn integration-status
  [project-id project db-spec]
  (jdbc/with-db-transaction
    [tx db-spec]
    (log/infof "Check code coverage app installation for project-id=%s." project-id)
    (let [repos (project.core/-repositories project tx)
          status (when-let [pr-config (project.core/-pr-integration-config project tx)]
                   (protocols/ci-installed? (pi/ci-provider pr-config) pr-config repos))]
      (log/infof "Check delta app installation FINISHED for project-id=%s %d repos." project-id (count repos))
      (mapv merge repos status))))