(ns codescene.features.code-coverage.parsers.dotcover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [clojure.set :as set]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage :statement-coverage])

(defn- covered-statement? [{:keys [attrs] :as _stmt-el}]
  (= "True" (:Covered attrs)))

(defn- uncovered-statement? [{:keys [attrs] :as _stmt-el}]
  (= "True" (:Covered attrs)))

(defn- statement->line [{:keys [attrs] :as _stmt-el}]
  (:Line attrs))

(defn statement-coverage [statement-els]
  (let [covered (filter covered-statement? statement-els)]
    (utils/->entry (count covered) (count statement-els))))

(defn- lines-matching 
  [covered-uncovered-fn? statements]
  (->> statements 
       (filter covered-uncovered-fn?)
       (map statement->line)
       set))

(defn line-coverage [parse-options statement-els]
  (let [lines (->> statement-els (map statement->line) distinct)
        covered   (lines-matching covered-statement? statement-els)
        uncovered (lines-matching uncovered-statement? statement-els)
        summary (utils/->entry (count covered) (count lines))]
    (merge summary
           (when (utils/needs-detailed-line-coverage? parse-options)
             {:details {:covered covered
                        :uncovered uncovered}}))))

(defn- parse-file-el [parse-options file->statements {:keys [attrs] :as file-el}]
  (let [{name :Name index :Index} attrs
        path (str/replace name "\\" "/")
        statements (file->statements index)]
    (m/assoc-some {:path path
                   :name (-> path (str/split #"/") last)}
                  :statement-coverage (statement-coverage statements)
                  :line-coverage (line-coverage parse-options statements))))

(defn- by-file-index
  [statements]
  (->> statements
       (group-by (comp :FileIndex :attrs))
       (into {})))

(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        files (xml-utils/sub-nodes-in xml [:FileIndices :File])
        file->statements (-> (xml-utils/sub-nodes-rec :Statement xml)
                             by-file-index)]
    (map (partial parse-file-el parse-options file->statements) files)))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "dotcover")
    (-name [this] "DotCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/dotCover.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
  )
