(ns codescene.features.api.code-coverage
  (:require [codescene.features.code-coverage.pr-check :as pr-check]
            [codescene.features.code-coverage.check.check-db :as check-db]
            [codescene.features.util.api :as api-utils]
            [codescene.features.api.core :as api-core]
            [codescene.features.code-coverage.parser :as parser]
            [codescene.features.code-coverage.parsers.core :refer [parser-matching-given]]
            [codescene.features.components.analytics :as analytics]
            [codescene.features.components.db :as db]
            [codescene.features.components.project :as comp-project]
            [codescene.features.components.code-coverage-storage :as storage]
            [codescene.url.url-utils :as url]
            [taoensso.timbre :as log]
            [clojure.java.io :as io]
            [clojure.string :as str])
  (:import (java.util.zip GZIPInputStream)
           (java.io PushbackInputStream InputStream)
           (org.apache.commons.io.input BOMInputStream)))

(defn- as-storable
  [{:keys [repo-url] :as metadata}]
  (merge {:repo (url/repo-url->repo-id repo-url)
          :subpath nil
          :repo-path nil}
         (dissoc metadata :repo-url)))

(defn- stored-metadata->event-properties
  [metadata]
  (select-keys metadata [:format :metric :repo]))

(defn request-upload
  [system user {:keys [format metric repo-url] :as metadata}]
  (let [parser (parser-matching-given format)
        metric (keyword metric)
        analytics (api-core/api-analytics system)]
    (cond
      (nil? parser)
      (api-utils/bad-request "Unsupported format")

      (let [supported? (set (parser/supported-metrics parser))]
        (not (supported? metric)))
      (api-utils/bad-request "Unsupported metric")

      (not (url/is-repo-url? repo-url))
      (api-utils/bad-request "Invalid repo-url")

      :else
      (let [storage (api-core/api-code-coverage-storage system)
            storable-metadata (as-storable metadata)
            upload-id (storage/request-upload storage user storable-metadata)
            ref (str (api-utils/code-coverage-url-prefix (api-core/api-root system)) "upload/" upload-id)]
        (analytics/send-user-event analytics user :code-coverage-upload-request (merge (stored-metadata->event-properties storable-metadata)
                                                                                       {:upload-id upload-id}))
        (api-utils/created {:ok "upload request accepted" :ref ref :id upload-id} {"Location" ref})))))

(defn- is-absolute-path?
  [path]
  ;; Handles both / and C:/ style paths
  ;; (but assumes forward slashes)
  (some? (re-find #"(^/)|(^[A-Z]:/)" path)))

(defn- has-absolute-paths?
  [coverage-data]
  (-> coverage-data first :path is-absolute-path?))

(defn- repo-path-does-not-match-absolute-paths
  [local-repo-path coverage-data]
  (let [p (-> coverage-data first :path)]
    (and (seq local-repo-path)
         (has-absolute-paths? coverage-data)
         (not (str/starts-with? p local-repo-path)))))

(defn- absolute-paths-but-no-repo-path
  [local-repo-path coverage-data]
  (and (empty? local-repo-path)
       (has-absolute-paths? coverage-data)))

(defn- metric-is-not-available
  [metric coverage-data]
  (let [metric (keyword metric)]
    (not (some #(get % metric) coverage-data))))

(defn ensure-ends-with
  [s substr]
  (cond-> s
    (not (str/ends-with? s substr)) (str substr)))

(defn remove-at-start
  [s substr]
  (cond-> s
    (str/starts-with? s substr) (subs (count substr))))

(defn- remove-local-repo-paths
  [local-repo-path coverage-data]
  (let [;; make sure there is a single '/' at the end
        local-repo-path (ensure-ends-with local-repo-path "/")
        remove-repo-path #(remove-at-start % local-repo-path)]
    (cond->> coverage-data
             (seq local-repo-path)
             (map #(update % :path remove-repo-path)))))

(defn- add-subpaths
  [subpath coverage-data]
  (let [with-subpath? (and (seq subpath) (not (has-absolute-paths? coverage-data)))
        ;; in case there is no provided subpath we still need an empty string
        ;; when the parser is jacoco the path might contain %s
        empty-or-subpath (cond-> "" with-subpath? (str subpath "/"))
        ;; the path may contain %s if the subpath need to be injected in the middle
        update-with #(if (str/includes? % "%s") (format % empty-or-subpath) (str empty-or-subpath %))]
    (map #(update % :path update-with) coverage-data)))

(defn- ->BomInputStream
  [s]
  (.. (BOMInputStream/builder) (setInputStream s) get))

(defn- may-be-gzip-input-stream
  "Check if stream starts with gzip magic bytes (0x1f, 0x8b) and wrap it into GZIPInputStream else return the original stream"
  [^InputStream in]
  (let [pbin (PushbackInputStream. in 2)]
      (let [byte1 (.read pbin)
            byte2 (.read pbin)]
        (when (>= byte2 0) (.unread pbin byte2))
        (when (>= byte1 0) (.unread pbin byte1))
        (if (and (= byte1 0x1f) (= byte2 0x8b))
          (GZIPInputStream. pbin)
          pbin))))

(defn- validate-and-parse [parser {:keys [metric repo-path subpath cli-command] :as _metadata} data]
  (with-open [r (-> data may-be-gzip-input-stream ->BomInputStream io/reader)]
    (let [repo-path (some-> repo-path (str/replace "\\" "/"))
          parse-options (when cli-command {:cli-command cli-command})
          coverage (doall (parser/read-coverage parser r parse-options))]
      (cond
        (empty? coverage)
        [false "The coverage data does not contains any records related to the current repo"]

        (repo-path-does-not-match-absolute-paths repo-path coverage)
        [false "Repo path does not match absolute paths in coverage data"]

        (absolute-paths-but-no-repo-path repo-path coverage)
        [false "Repo path needs to be specified when coverage data contains absolute paths"]

        (metric-is-not-available metric coverage)
        [false "Requested metric is not present in coverage data"]

        :else
        [true (->> coverage
                   (remove-local-repo-paths repo-path)
                   (add-subpaths subpath))]))))

(defn parse-coverage-with
  [p metadata data]
  (try
    (validate-and-parse p metadata data)
    (catch Exception e
      (log/error e "Failed to parse the uploaded coverage data")
      [false (str "Failed to parse the uploaded coverage data: " (.getMessage e))])))

(defn parse-coverage
  [{:keys [format] :as metadata} data]
  (if-let [p (parser-matching-given format)]
    (parse-coverage-with p metadata data)
    [false (str "Invalid format: " format)]))

(defn upload-data
  [system user upload-id data]
  (let [storage (api-core/api-code-coverage-storage system)
        {:keys [format data-path] :as metadata} (storage/metadata storage upload-id)
        analytics (api-core/api-analytics system)]
    (cond
      (or (nil? metadata) (some? data-path)) ;; don't allow multiple uploads attempts to the same id
      (api-utils/bad-request "Invalid upload id")

      :else
      (do
        (analytics/send-user-event analytics user :code-coverage-upload (merge (stored-metadata->event-properties metadata)
                                                                               {:upload-id upload-id}))
        (let [[ok? coverage-or-error] (parse-coverage metadata data)]
          (if ok?
            (let [msg (str "Successfully parsed " format " data for " (count coverage-or-error) " files.")]
              (storage/upload-data storage user upload-id coverage-or-error)
              (log/info msg)
              (analytics/send-user-event analytics user :code-coverage-upload-success (merge (stored-metadata->event-properties metadata)
                                                                                             {:upload-id upload-id
                                                                                              :n-coverage (count coverage-or-error)}))
              (api-utils/ok {:ok msg}))
            (do
              (storage/abort-upload storage upload-id)
              (analytics/send-user-event analytics user :code-coverage-upload-failure (merge (stored-metadata->event-properties metadata)
                                                                                             {:upload-id upload-id
                                                                                              :error {:message coverage-or-error}}))
              (api-utils/bad-request coverage-or-error))))))))

; @CodeScene(disable: "Code Duplication") -- really annoying that this pops up :/
(defn delete-data
  [system user {:keys [repo-url] :as _query-params}]
  (cond
    (not (url/is-repo-url? repo-url))
    (api-utils/bad-request "Invalid repo-url")

    :else
    (let [storage (api-core/api-code-coverage-storage system)
          n-deleted (storage/delete-data storage user {:repo (url/repo-url->repo-id repo-url)})
          msg (str "Deleted " n-deleted " coverage data entries.")]
      (log/info msg)
      (api-utils/ok {:ok msg}))))

(defn get-data
  [system user {:keys [repo-url] :as _query-params}]
  (cond
    (not (url/is-repo-url? repo-url))
    (api-utils/bad-request "Invalid repo-url")

    :else
    (let [storage (api-core/api-code-coverage-storage system)
          data (storage/query-data storage user {:repo (url/repo-url->repo-id repo-url)})]
      (api-utils/ok data))))

(comment
  (let [parser (parser-matching-given "jacoco")
        data (io/input-stream (io/file "/Users/jumar/Downloads/jacoco.xml.gz"))
        parse-options {}
        coverage (parser/read-coverage parser (-> data GZIPInputStream. ->BomInputStream io/reader) parse-options)]
    (def my-coverage coverage))
  )

(defn get-project-config
  [system project-id]
  (let [coverage (api-core/api-coverage-check system)]
    (storage/project-settings coverage project-id)))

(defn check-configured
  [system project-id repo-url]
  (let [key-fn #(select-keys % [:repo-slug :owner-login :host])
        repo-info (key-fn (url/repo-url->repo-info repo-url false))]
    (some (fn [{:keys [url]}]
            (let [repo (url/repo-url->repo-info url false)]
              (when (= (key-fn repo) repo-info)
                repo)))
          (comp-project/all-repositories (api-core/api-projects system) project-id))))

(defn- check-result*
  [system project-id results]
  (when (:base-ref results)
    (storage/post-results (api-core/api-coverage-check system) project-id results))
  (check-db/update-with-results (db/db-spec system) project-id results))

(defn- event-data [system project-id results]
  (merge (select-keys (get-project-config system project-id)
                      [:gates :project-id])
         {:result (assoc (select-keys results [:repo-id :commit-sha :files-count :duration-ms])
                    :outcome (pr-check/check-status (:coverage-results results)))}))

(defn- event-type [results]
  (if (:base-ref results)
    :coverage-pr-check
    :coverage-check))

(defn check-result
  [system user project-id {:keys [repo-url] :as results}]
  (let [analytics (api-core/api-analytics system)
        results (-> results
                    (dissoc :repo-url)
                    (update :base-ref not-empty)
                    (assoc :repo-id (url/repo-url->repo-id repo-url)))
        event-base (event-data system project-id results)]
    (try
      (if (check-configured system project-id repo-url)
        (do (check-result* system project-id results)
            (analytics/send-user-event analytics
                                       user
                                       (event-type results)
                                       event-base)
            (api-utils/ok {:ok "ok"}))
        (do (log/infof "PR Coverage check results ignored: repo %s not configured in project %s"
                       repo-url project-id)
            (api-utils/not-found (format "Project %d doesn't have repository %s" project-id repo-url))))
      (catch Exception e
        (log/info e "PR Coverage check failed")
        (case (-> e ex-data :type)
          :not-found (api-utils/not-found (ex-message e))
          (do (analytics/send-user-event analytics user :coverage-check-error
                                         (assoc event-base :msg (ex-message e)))
              (api-utils/internal-server-error (ex-message e))))))))
