(ns clj-codescene-plugin.setting
  (:import
   (com.codescene.plugin.configuration Option
                                       Setting
                                       SettingType
                                       SettingValue
                                       ListSetting
                                       MultiSelectSetting
                                       PasswordSetting
                                       TextSetting
                                       CheckboxSetting)))

(defn SettingType->str [type]
  (case (.name type)
    "CHECKBOX" "checkbox"
    "TEXT" "text"
    "PASSWORD" "password"
    "SELECT" "select"
    "MULTISELECT" "multi-select"
    "LIST" "list"))

(defn str->SettingType [type]
  (case type
    "checkbox" SettingType/CHECKBOX
    "text" SettingType/TEXT
    "password" SettingType/PASSWORD
    "select" SettingType/SELECT
    "multi-select" SettingType/MULTISELECT
    "list" SettingType/LIST))

(defn encrypted-setting-type? [type]
  (#{"password"} type))

(defn list-setting-type? [type]
  (#{"multi-select","list"} type))

(defn option-setting-type? [type]
  (#{"multi-select","select"} type))

(defn ->Setting [{:keys [type id name caption default-value options] :as _setting}]
  (reify Setting
    (getType [_] (str->SettingType type))
    (getId [_] id)
    (getName [_] name)
    (getCaption [_] caption)
    (getDefaultValue [_] default-value)
    (getOptions [_] (if (option-setting-type? type)
                      options
                      (throw (UnsupportedOperationException.))))))

(defn ->TextSetting [{:keys [id name caption default-value]}]
  (proxy [TextSetting] []
    (getId [] id)
    (getName [] name)
    (getCaption [] (or caption ""))
    (getDefaultValue [] (or default-value []))))

(defn ->Option [{:keys [name key]}]
  (reify Option
    (getName [_] name)
    (getKey [_] key)))

(defn ->MultiSelectSetting [{:keys [id name caption default-value options]}]
  (proxy [MultiSelectSetting] []
    (getId [] id)
    (getName [] name)
    (getCaption [] (or caption ""))
    (getDefaultValue [] (or default-value []))
    (getOptions [] (map ->Option options))))

(defn ->ListSetting [{:keys [id name caption default-value]}]
  (proxy [ListSetting] []
    (getId [] id)
    (getName [] name)
    (getCaption [] (or caption ""))
    (getDefaultValue [] (or default-value []))))

(defn ->PasswordSetting [{:keys [id name caption default-value]}]
  (proxy [PasswordSetting] []
    (getId [] id)
    (getName [] name)
    (getCaption [] (or caption ""))
    (getDefaultValue [] (or default-value []))))

(defn ->CheckboxSetting [{:keys [id name caption default-value]}]
  (proxy [CheckboxSetting] []
    (getId [] id)
    (getName [] name)
    (getCaption [] (or caption ""))
    (getDefaultValue [] (or default-value []))))

(defn ->SettingValue [{:keys [id value] :as _value}]
  (reify SettingValue
    (getId [_] id)
    (getValue [_] (or value []))))

(defn Option->map [option]
  {:key (.getKey option)
   :name (.getName option)})

(defn options [type setting]
  (when (option-setting-type? type)
    (map Option->map (.getOptions setting))))

(defn Setting->map [setting]
  (let [type (->> (.getType setting) SettingType->str)]
    (merge
     {:name (.getName setting)
      :id (.getId setting)
      :type type
      :default-value (.getDefaultValue setting)
      :caption (.getCaption setting)}
     (when (option-setting-type? type)
       {:options (options type setting)}))))

(defn SettingValue->map [setting-value]
  {:id (.getId setting-value)
   :value (.getValue setting-value)})

(defn ->value-lookup [setting-values]
  (->> setting-values
       (map SettingValue->map)
       (map (juxt :id :value))
       (into {})))