(ns clj-codescene-plugin.metric
  (:import
   (com.codescene.plugin.systemmap MetricType
                                   BooleanMetric
                                   FloatMetric
                                   EnumMetric
                                   EnumValue
                                   IntMetric
                                   OptionalValue)))

(defn ->IntMetric [{:keys [id name unit min-value max-value color is-positive is-summable]}]
  (proxy  [IntMetric] []
    (getId [] id)
    (getName [] name)
    (getUnit [] (or unit (proxy-super getUnit)))
    (getMinValue [] (if min-value (OptionalValue/ofInt min-value) (proxy-super getMinValue)))
    (getMaxValue [] (if max-value (OptionalValue/ofInt max-value) (proxy-super getMaxValue)))
    (getColor [] (or color (proxy-super getColor)))
    (isPositive [] (if (some? is-positive) is-positive (proxy-super isPositive)))
    (isSummable [] (if (some? is-summable) is-summable (proxy-super isSummable)))))

(defn ->FloatMetric [{:keys [id name unit min-value max-value color is-positive is-summable]}]
  (proxy  [FloatMetric] []
    (getId [] id)
    (getName [] name)
    (getUnit [] (or unit (proxy-super getUnit)))
    (getMinValue [] (if min-value (OptionalValue/ofFloat min-value) (proxy-super getMinValue)))
    (getMaxValue [] (if max-value (OptionalValue/ofFloat max-value) (proxy-super getMaxValue)))
    (getColor [] (or color (proxy-super getColor)))
    (isPositive [] (if (some? is-positive) is-positive (proxy-super isPositive)))
    (isSummable [] (if (some? is-summable) is-summable (proxy-super isSummable)))))

(defn ->BooleanMetric [{:keys [id name]}]
  (proxy  [BooleanMetric] []
    (getId [] id)
    (getName [] name)))

(defn ->EnumValue [{:keys [name color]}]
  (reify EnumValue
    (getName [_] name)
    (getColor [_] color)))

(defn ->EnumMetric [{:keys [id name values]}]
  (proxy  [EnumMetric] []
    (getId [] id)
    (getName [] name)
    (getValues [] (map ->EnumValue values))))

(defn MetricType->str [type]
  (case (.name type)
    "BOOLEAN" "boolean"
    "ENUM" "enum"
    "INT" "int"
    "FLOAT" "float"))

(defn str->MetricType [type]
  (case type
    "boolean" MetricType/BOOLEAN
    "enum" MetricType/ENUM
    "int" MetricType/INT
    "float" MetricType/FLOAT))

(defn- EnumValue->map [value]
  {:name (.getName value)
   :color (.getColor value)})

(defn- interpret-value [type value]
  (when (and value (.isPresent value))
    (case type
      "boolean" (.getAsBoolean value)
      "float" (.getAsFloat value)
      (.getAsInt value))))

(defn- type-specific-props [type metric]
  (case type
    "enum" {:values (mapv EnumValue->map (.getValues metric))}
    "boolean" {}
    {:max-value (interpret-value type (.getMaxValue metric))
     :min-value (interpret-value type (.getMinValue metric))
     :color (interpret-value "int" (.getColor metric))
     :unit (.getUnit metric)
     :is-positive (.isPositive metric)
     :is-summable (.isSummable metric)}))

(defn Metric->map [metric]
  (let [id (.getId metric)
        name (.getName metric)
        type (MetricType->str (.getType metric))]
    (merge {:name name
            :id id
            :type type}
           (type-specific-props type metric))))

(defn value [metric value]
  (interpret-value (:type metric) value))