// @flow
import $ from 'jquery';
import d3 from 'd3';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as brooks from 'codescene-ui/src/analyses/brooks-law';
import * as authorTrend from 'codescene-ui/src/analyses/author-experience-trend';
import * as alerts from 'codescene-ui/src/alerts';

type ChurnTrendViewOptions = {
  revisionTrendUrl : string,
  authorsPerMonthTrendUrl : string,
  authorsExperienceTrend : string
};

const parseDate = d3.time.format('%Y-%m-%d').parse;

function parseTotalExperience(row) {
  var parsed = {
    date: parseDate(row.date),
    total: 0
  };
  const nauthors = parseInt(row.nauthors);
  Object.keys(row)
    .filter(c => c === 'onboarded' || c === 'experienced' || c === 'veterans')
    .forEach(key => {
      const value = parseInt(row[key]);

      parsed[key] = nauthors * (value * 0.01);
    });
  parsed.total = nauthors;
  parsed.totalexperience = parseInt(row.totalexperience);
  parsed.meanexperience = row.meanexperience ? parseInt(row.meanexperience) : 0;
  parsed.weightedexperience = row.weightedexperience ? parseInt(row.weightedexperience) : 0;
  return parsed;
}

export function init(options : ChurnTrendViewOptions) {

  xhr.csv(options.revisionTrendUrl).then(data => {
    const trendContainer = $('.brooks-law-graph');
    xhr.csv(options.authorsPerMonthTrendUrl).then(activeAuthors => {
      if (activeAuthors === undefined || activeAuthors.length == 0) {
        alerts.warning(trendContainer, '<strong>Result not available.</strong> This is a new type of analysis so you need to re-run your analysis to get the results.');
      } else {
        brooks.display($('.brooks-law-graph'), data, activeAuthors);
      }
    })
      .catch(err => {
        console.error(err);
        alerts.warning(trendContainer, '<strong>Result not available.</strong> This is a new type of analysis so you need to re-run your analysis to get the results.');
      });
  });


  const authorTrendContainer = $('.author-experience-graph');
  xhr.csv(options.authorsExperienceTrend, parseTotalExperience).then(data => {
    authorTrend.displayTrendFrom(authorTrendContainer, data, 2);
  })
    .catch(err => {
      console.error(err);
      alerts.warning(authorTrendContainer, '<strong>No Analysis Data:</strong> The Author Experience Trends are not available. Ensure you have enabled the complete Git history for knowledge metrics in the project configuration. Also, since this is a new analysis, re-run the analysis to generate the trends.');
    });
}


export function dashboardInit(options : ChurnTrendViewOptions) {
  const revisionTrendPromise = xhr.csv(options.revisionTrendUrl);
  const activeAuthorsPromise = xhr.csv(options.authorsPerMonthTrendUrl);

  const noDataMessage = '<strong>Results not available.</strong> This is a new type of analysis so you need to re-run your analysis to get the results.';
  const trendContainer = $('.brooks-law-graph');
  
  Promise.all([revisionTrendPromise, activeAuthorsPromise])
    .then((results) => {
      var data, activeAuthors;
      [data, activeAuthors] = results;
      if (activeAuthors === undefined || activeAuthors.length == 0) {
        alerts.warning(trendContainer, noDataMessage);
        trendContainer.parent().addClass('no-brooks-data');
      }
      else {
        brooks.dashboardDisplay(trendContainer, data, activeAuthors);
      }
    })
    .catch(() => {
      alerts.warning(trendContainer, noDataMessage);
      trendContainer.parent().addClass('no-brooks-data');
    });
}

type BrooksByTeamOptions = {
  teamTrendRootUrl: string,
  teamName: string,
  uiElement: string
};

export function initTeamView(options : BrooksByTeamOptions) {
  const teamSpecificUrl = options.teamTrendRootUrl + '?team=' + options.teamName; // already escaped
  const trendContainer = $(options.uiElement);

  xhr.json(teamSpecificUrl)
    .then(data => {
      if (data === undefined || data.length == 0) {
        alerts.warning(trendContainer, '<strong>Result not available.</strong> We do not have any data for this team.');
      } else {
        brooks.displayWithWeeklyKnownAuthors(trendContainer, data);
      }
    })
    .catch(err => {
      console.error(err);
      alerts.warning(trendContainer, '<strong>Failed to load results.</strong> Check the browser console for possible error messages and report to Empear.');
    });
}
