grammar CSharpSharedRulesParser;

import CSharpSharedRulesLexer;

translationunit
:
	expression* EOF
;

expression : function_declaration
           | property_with_object_initialization
           | pre_proc_top_level_start
           | namespace_block
           | class_declaration
           | block_statement
           | anything;

anything : ~(LeftBrace | RightBrace);

namespace_block : NAMESPACE (ID | SCOPED_NAME) LeftBrace expression* RightBrace;

property_with_object_initialization : '=' 'new' ID LeftParen function_definition_params_list? RightParen LeftBrace function_body RightBrace;

class_declaration : (CLASS | STRUCT | RECORD) class_name primary_constructor_args? ~(LeftBrace)*? LeftBrace expression* RightBrace;
primary_constructor_args: LeftParen function_definition_params_list RightParen;
class_name : ID;

// We just want to get rid of the symbol following the #ifdef so that it's not part of a possible subsequent function.
pre_proc_top_level_start : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_top_level_condition;

// We use a different rule on the top-level since we need to parse more carefully (and significantly slower) in
// order to not consume function signatures that may follow immediately after.
pre_proc_top_level_condition : '!'?  ('1' | '0' | ID)
                             | LeftParen pre_proc_top_level_condition RightParen
                             | 'defined' LeftParen pre_proc_top_level_condition RightParen;

function_declaration : function_name generic_return_spec? LeftParen function_definition_params_list? RightParen generic_type_constraints? ':' member_init_list full_method_definition
                     | function_name generic_return_spec? LeftParen function_definition_params_list? RightParen generic_type_constraints? full_method_definition
                     | function_type_signature function_name generic_return_spec? LeftParen function_definition_params_list? RightParen generic_type_constraints? full_method_definition;
function_type_signature : ~(LeftBrace | RightBrace | LeftParen | RightParen |
                            PRE_PROC_IFDEF | PRE_PROC_IF | PRE_PROC_IFNDEF |
                            PRE_PROC_ENDIF | PRE_PROC_ELSE | SEMICOLON | '=' | CLASS | ',' | 'new')+?;
function_name : '~'? (ID | SCOPED_NAME);
generic_return_spec: generic_type;

generic_type: '<' generic_arg_list '>';
generic_arg: ~('<' | '>')*? generic_type?;
generic_arg_list: generic_arg
                 | generic_arg_list ',' generic_arg;

member_init_list : member_init_name LeftParen ~(LeftBrace)*;
member_init_name : 'base' | 'this';

fun_arg : modifier* (generic_args | parameter) default_value?;

modifier: 'ref' | 'out' | 'in' | 'params' | 'this'
        | metadata_argument_modifier;

metadata_argument_modifier: '[' ~(LeftBrace | ARROW_OPERATOR | ';')+? ']';

nullable: '?';

default_value: '=' ~(',' | LeftParen | RightParen | ';')+? (LeftParen ~(RightParen)*? RightParen)?;

type_name: (ID | SCOPED_NAME);

generic_args: type_name generic_type  nullable? ID;

argument_type: type_name nullable? '[]'?;
parameter: argument_type  nullable? ID;

function_definition_params_list : fun_arg
                                | function_definition_params_list ',' fun_arg;

// C# allows constraints on generics. For example:
//  void SwapIfGreater<T>(ref T lhs, ref T rhs) where T : System.IComparable<T>
generic_type_constraints: 'where' ~(ARROW_OPERATOR | LeftBrace | ';' | PUBLIC | PRIVATE | PROTECTED | ABSTRACT | INTERNAL)+?;

full_method_definition: (LeftBrace function_body RightBrace)
                      | expression_body_definition;

expression_body_definition: ARROW_OPERATOR function_body ';';

function_body : function_body_statement*?;

function_body_statement : pre_proc_block
                        | new_object_expr // important order or precedence: must come before any local function
                        | local_function
                        | block_statement
                        | any_statement;

// We cannot have local constructors, so always require a return type.
// Without this additional strictness, we risk parsing stuff like:
//  catch (ArgumentException e) { }
local_function: local_function_return_type function_declaration;
local_function_return_type: type_name | '>' | ']';

// we need to filter these out, or the resulting expression might look just like a local function (constructor).
// Consider:
//  new Dictionary<string, object>() { { "input", inputString } })
new_object_expr: 'new' (ID | SCOPED_NAME);

block_statement : LeftBrace function_body_statement*? RightBrace;

pre_proc_block : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_condition? pre_proc_block_statement*? (PRE_PROC_ELSE | PRE_PROC_ENDIF);
pre_proc_block_statement : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_condition? pre_proc_block_statement*? PRE_PROC_ENDIF
                         | ~(PRE_PROC_ENDIF | PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF);
pre_proc_condition : ('!' | '1' | '0' | ID);

any_statement : ~(LeftBrace | RightBrace | CLASS | NAMESPACE | PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF);
