(ns codescene.features.util.template
  (:require [cheshire.factory :as json-factory]
            [selmer.parser :as selmer])
  (:import (com.fasterxml.jackson.core.io CharacterEscapes)))

(def html-template-character-escapes
  "These escapes are used when encoding JSON inside a <script> tag in a template.

  This prevents a string value in JSON closing the <script> tag.

  Backslash is also changed to unicode escape to counter 'escape the escape' style attacks."
  (let [escapes (doto (CharacterEscapes/standardAsciiEscapesForJSON)
                  ; escape " as \u0022 instead of \", also escape < >
                  (aset (int \") CharacterEscapes/ESCAPE_STANDARD)
                  (aset (int \<) CharacterEscapes/ESCAPE_STANDARD)
                  (aset (int \>) CharacterEscapes/ESCAPE_STANDARD)
                  (aset (int \\) CharacterEscapes/ESCAPE_STANDARD))]
    (proxy [CharacterEscapes] []
      (getEscapeCodesForAscii [] escapes)
      (getEscapeSequence [_v] nil))))

(def escaping-json-factory
  "JSON Factory with special encoding rules to make json Selmer filter emit
  JSON strings that have certain HTML characters escaped with unicode escape sequences."
  (doto (json-factory/make-json-factory json-factory/default-factory-options)
    (.setCharacterEscapes html-template-character-escapes)))

(defn render-file [filename-or-url context-map & [opts]]
  (binding [json-factory/*json-factory* escaping-json-factory]
    (if opts
      (selmer/render-file filename-or-url context-map opts)
      (selmer/render-file filename-or-url context-map))))

(comment
  (selmer/render-file "templates/error.html" {})
  (selmer/render "{% now \"dd-MM-yyyy HH:MM:ss\"%}" {})
  ;; => "\"08-05-2024 06:05:49\""
  )
