(ns codescene.features.reports.common.project-card
  (:require
   [codescene.features.reports.common.pdf-elements :as pdf-elements]
   [codescene.features.reports.pdf-helpers :as ph]
   [codescene.features.reports.common.styling :as styling]
   [codescene.features.presentation.scoring :as scoring]))


(defn- factor-content [{:keys [score kpitrend]} img-dir {:keys [image-styles trend-image-styles trend-paragraph-styles]}]
  (let [score-icon (styling/score->icon img-dir score)
        [text image-file] (cond
                            (or (nil? kpitrend) (zero? kpitrend)) ["No trend" "No-trend.png"]
                            (< 0 kpitrend) ["Improving" "Improving.png"]
                            (> 0 kpitrend) ["Declining" "Declining.png"])
        image (styling/load-resource img-dir image-file)]
    (ph/table
     [] [3 5]
     (ph/row
      (ph/cell [] (ph/image image-styles score-icon))
      (ph/cell [] (ph/table [] [1 4] (ph/row
                                      (ph/cell [] (ph/image trend-image-styles image))
                                      (ph/cell trend-paragraph-styles text))))))))


(defn- factor [title trend-content {:keys [cell-styles title-cell-styles title-paragraph-styles content-cell-styles]}]
  (ph/cell
   cell-styles
   (ph/table
    [] nil
    (ph/row
     (ph/cell title-cell-styles (ph/paragraph title-paragraph-styles title)))
    (ph/row
     (ph/cell
      content-cell-styles
      trend-content)))))


(defn- factor-value
  "Digs out the first and last value of the trend (using the :value displayed in the trend graphs)
   and also the score calculated by scoring the weighted kpi of the last sample."
  [trend factor-key]
  (let [last-sample (last trend)
        last-value (get last-sample :value 0)
        first-value (get (first trend) :value 0)
        kpitrend (- last-value first-value)
        score (:score (scoring/score-info-for (get last-sample :weighted-kpi
                                                   (:value last-sample)) factor-key))]
    {:kpitrend kpitrend
     :score score}))


(defn- factor-values [{:keys [code-health knowledge-distribution team-code-alignment delivery-performance]}]
  (let [ch-values (factor-value code-health :code-health)
        kd-values (factor-value knowledge-distribution :knowledge-distribution)
        tca-values (factor-value team-code-alignment :team-code-alignment)
        dp-values (factor-value delivery-performance :delivery-performance)]
    {:ch-values ch-values
     :kd-values kd-values
     :tca-values tca-values
     :dp-values dp-values}))


(defn pdf-element-code-health
  [renderable-trends img-dir]
  (let [{:keys [ch-values kd-values tca-values dp-values]} (factor-values renderable-trends)
        main-styles {:title-cell-styles [:c-factor-subheader-cell]
                :title-paragraph-styles [:c-factor-subheader-paragraph]
                :content-cell-styles [:c-factor-contents-cell]}
        content-styles {:image-styles [:c-health-image]
                        :trend-image-styles [:c-trend-image]
                        :trend-paragraph-styles [:c-subheader]}]
    (ph/table [] nil (ph/row
                      (factor "Code Health"
                              (factor-content ch-values img-dir content-styles)
                              (assoc main-styles :cell-styles [:c-factor-cell (styling/score->cell-style (:score ch-values))]))
                      (factor "Knowledge Distribution"
                              (factor-content kd-values img-dir content-styles)
                              (assoc main-styles :cell-styles [:c-factor-cell (styling/score->cell-style (:score kd-values))]))
                      (factor "Team-Code Alignment"
                              (factor-content tca-values img-dir content-styles)
                              (assoc main-styles :cell-styles [:c-factor-cell (styling/score->cell-style (:score tca-values))]))
                      (factor "Delivery"
                              (factor-content dp-values img-dir content-styles)
                              (assoc main-styles :cell-styles [:c-factor-cell (styling/score->cell-style (:score dp-values))]))))))

(defn pdf-element-portfolio
  [renderable-trends {:keys [pm-conf-url teams-conf-url]} img-dir]
  (let [{:keys [ch-values kd-values tca-values dp-values]} (factor-values renderable-trends)
        main-styles {:cell-styles [:c-factor-small-cell]
                :title-cell-styles [:c-factor-small-subheader-cell]
                :title-paragraph-styles [:c-factor-small-subheader-paragraph]
                :content-cell-styles [:c-factor-small-contents-cell]}
        content-styles {:image-styles [:c-factor-small-image]
                        :trend-image-styles [:c-factor-small-trend-image]
                        :trend-paragraph-styles [:c-factor-small-trend-paragraph]}]
    (ph/table [] nil (ph/row
                      (factor "Code Health"
                              (factor-content ch-values img-dir content-styles)
                              main-styles)
                      (factor "Knowledge Distribution"
                              (factor-content kd-values img-dir content-styles)
                              main-styles)
                      (factor "Team-Code Alignment"
                              (if (= (:score tca-values) -1)
                                (ph/table [] [15 7] (pdf-elements/link-row-small teams-conf-url "Configure Teams" img-dir))
                                (factor-content tca-values img-dir content-styles))
                              main-styles)
                      (factor "Delivery"
                              (if (= (:score dp-values) -1)
                                (ph/table [] [18 7] (pdf-elements/link-row-small pm-conf-url "Configure PM tool" img-dir))
                                (factor-content dp-values img-dir content-styles))
                              main-styles)))))

