(ns codescene.features.reports.common.project
  (:require [codescene.factors.core :as factors]
            [codescene.features.components.analysis :as analysis]
            [codescene.features.components.navigation :as navigation]
            [codescene.features.components.project :as project]
            [codescene.features.factors.results :as results]
            [codescene.features.presentation.time-scope :as ts]
            [medley.core :as m]
            [tick.core :as t]))

(defn path-fn [system project-id latest-successful-analysis]
  (let [analysis-component (:analysis-component (:codescene.features.reports/report-component system))]
    (analysis/local-analysis-result-path analysis-component project-id latest-successful-analysis)))

(defn with-analysis-info [system {:keys [id] :as project}]
  (let [analysis-component (:analysis-component (:codescene.features.reports/report-component system))
        {analysis-id :id :as analysis} (analysis/latest-successful-analysis analysis-component id)
        path-fn (when analysis-id (analysis/local-analysis-result-path analysis-component id analysis-id))]
    (merge project {:path-fn path-fn 
                    :analysis-id analysis-id
                    :analysis analysis})))

(defn project-name [system project-id]
  (let [project-component (:project-component (:codescene.features.reports/report-component system))]
    (:name (project/project-by-id project-component project-id))))

(defn periods [absolute-analysis-path]
  (let [end-date (results/end-date absolute-analysis-path)
        [monthago _] (ts/interval-from-scope end-date "month")
        [yearago _] (ts/interval-from-scope end-date "year")
        prev-end-date (t/<< monthago (t/new-period 1 :days))
        [prev-monthago _] (ts/interval-from-scope prev-end-date "month")]
    {:yearago yearago
     :prev-monthago prev-monthago
     :prev-end-date prev-end-date
     :monthago monthago
     :end-date (t/>> end-date (t/new-period 1 :days))}))

(defn- scoped-delivery-performance [delivery-performance-trend scope]
  (map #(merge % (scope (:kpis %))) delivery-performance-trend))

(defn clean-renderable-trends [renderable-trends delivery-performance-scope]
  (let [factors (merge (select-keys renderable-trends factors/factor-keys)
                       {:delivery-performance
                        (scoped-delivery-performance (:delivery-performance renderable-trends) delivery-performance-scope)})]
    (m/map-vals (fn [trend] (remove #(or (contains? % :error) (contains? % :info)) trend)) factors)))

(defn generate-project-links [system project-id analysis-id]
  (when analysis-id
    (let [navigation-component (:navigation-component (:codescene.features.reports/report-component system))
          http-base (navigation/get-http-base navigation-component)
          analysis-results-uri (navigation/get-analysis-results-uri navigation-component project-id analysis-id)
          pr-stats-uri (navigation/get-pr-statistics-uri navigation-component project-id analysis-id)
          pr-conf-uri (navigation/get-project-conf-pr-integration-uri navigation-component project-id)
          pm-conf-uri (navigation/get-project-conf-pm-integration-uri navigation-component project-id)
          teams-conf-uri (navigation/get-project-conf-teams-uri navigation-component project-id)
          alerts-uri-fn (partial navigation/get-warning-uri navigation-component project-id analysis-id)]
      {:http-base http-base
       :dashboard-url (str http-base analysis-results-uri)
       :pr-stats-url (str http-base pr-stats-uri)
       :pr-conf-url (str http-base pr-conf-uri)
       :pm-conf-url (str http-base pm-conf-uri)
       :teams-conf-url (str http-base teams-conf-uri)
       :alerts-uri-fn alerts-uri-fn})))

