(ns codescene.features.recommendations.presentation
  "Prepare event log data (extracted in `event-log/event-analysis`) for presentation in the UI."
  (:require
   [codescene.features.recommendations.event-log.event-analysis :as event-analysis]
   [codescene.features.recommendations.definitions.recommendations :as recommendations]
   [codescene.features.recommendations.definitions.predicates :as predicates]))

;;; The goal is to produce something like this:

;; [
;;   {
;;     "category": {
;;       "categoryType": "improve",
;;       "icon": "warn",
;;       "id": "manage-risk",
;;       "label": "Manange risk"
;;     },
;;     "id": "stopdcl",
;;     "initTime": "2024-02-29",
;;     "issuesParagraphs": [
;;       "6 files decline in Code Health, meaning they are now more expensive to maintain than they used to be."
;;     ],
;;     "opportunityParagraphs": [
;;       "No matter the baseline, you never want your code to get worse and to take on additional technical debt.",
;;       "By implementing “Quality Gates” for Hotspots you can improve and maintain a high level of Code Health."
;;     ],
;;     "progressItems": [
;;       {
;;         "status": "complete",
;;         "statusText": "2024-03-28",
;;         "title": "Set quality gates on 6 of the 6 files (6/6)"
;;       },
;;       {
;;         "status": "in-progress",
;;         "statusText": "2024-03-28",
;;         "title": "Goal violation on 1 of the 6 files"
;;       }
;;     ],
;;     "progressPercent": 85,
;;     "title": "Stop the code health decline"
;;   },
;;   {
;;     "category": {
;;       "categoryType": "risk",
;;       "icon": "coin",
;;       "id": "technical-debt",
;;       "label": "Technical debt"
;;     },
;;     "id": "confpr",
;;     "initTime": "2024-02-29",
;;     "issuesParagraphs": [
;;       "CodeScene's PR integration is not configured for 3 of the four repositories in this project."
;;     ],
;;     "opportunityParagraphs": [
;;       "Feedback on pull requests helps developers prevent technical debt from creeping back in to projects.",
;;       "Setting up the PR integration is easy and is the first step when trying to improve the Code Health of a project."
;;     ],
;;     "progressItems": [
;;       {
;;         "status": "in-progress",
;;         "statusText": "2024-02-19",
;;         "title": "Configured PR integration on 1 of the 4 repositories"
;;       }
;;     ],
;;     "progressPercent": 25,
;;     "title": "Enable automated code review"
;;   }
;; ]


(defn- predicate-tuple->map
  [[stream event identifier user-id]]
  (cond-> {:stream stream :event event}
    identifier (assoc :identifier identifier)
    user-id (assoc :user-id user-id)))

(defn- progress-percent
  [resolved-count unresolved-count]
  (if (pos? (+ resolved-count unresolved-count))
    (double (* 100 (/ resolved-count (+ resolved-count unresolved-count))))
    0))                                 ;or should this be 100? 🤷

(defn- issue-paragraphs
  "Actually just one paragraph for now."
  [{:keys [issue-formats] :as _recommendation} recommendation-event]
  (let [{:keys [singular multiple]} issue-formats
        {:keys [identifiers]} recommendation-event]
    (cond
      (and multiple (> (count identifiers) 1))
      [(format multiple (count identifiers))]

      singular
      [singular]

      :else ; no formats
      [])))


(defn predicate-statuses-grouped-by-predicate
  "Uses the stream and event values to group the predicates. This way,
  all the items with the same predicate but different identifiers will
  get grouped together. From there, we can calculate progress."
  [predicate-statuses]
  (group-by (fn [[k _v]] (vec (take 2 k))) predicate-statuses))

(defn- get-predicate-format
  [formats kw]
  (get formats kw
       (get {:multiple "%d complete out of %d"
             :singular "Complete"
             :unresolved-singular "Incomplete"
             :unresolved-multiple "No items complete"}
            kw)))

(defn- interpreted-predicate-text
  [event-kw resolved-count unresolved-count]
  (let [formats (:progress-format (predicates/event-keyword->predicate event-kw))]
    (cond (and (zero? resolved-count) (> unresolved-count 1))
          (get-predicate-format formats :unresolved-multiple)

          (> (+ resolved-count unresolved-count) 1)
          (format (get-predicate-format formats :multiple) resolved-count (+ resolved-count unresolved-count))

          (= 1 resolved-count)
          (get-predicate-format formats :singular)

          (= 1 unresolved-count)
          (get-predicate-format formats :unresolved-singular))))

(defn human-readable-progress-report
  "Takes a list of predicate-statuses for the same stream/event and
  produces a report string."
  [predicate-statuses]
  (let [event (second (ffirst predicate-statuses))
        resolved-count (count (filter (fn [[k v]] (= :resolved (:status v))) predicate-statuses))
        unresolved-count (count (filter (fn [[k v]] (= :unresolved (:status v))) predicate-statuses))]
    (interpreted-predicate-text event resolved-count unresolved-count)))

(defn- predicate-statuses-summary-string
  [predicate-statuses]
  (cond (every? #(= :resolved (:status (second %))) predicate-statuses)
        "complete"
        (every? #(= :unresolved (:status (second  %))) predicate-statuses)
        "notStarted"
        :else
        "inProgress"))

(defn progress-items
  [predicate-statuses]
  (->>  predicate-statuses
        predicate-statuses-grouped-by-predicate
        (map (fn [[[_stream event] vs]]
               {:title (human-readable-progress-report vs)
                :status (predicate-statuses-summary-string vs)}))))

(comment
  (def a {[:analysis :goal-ok "a" nil]
          {:status :resolved,
           :event {:stream :analysis, :event :goal-ok, :identifier "a"}}
          [:analysis :goal-ok "b" nil]
          {:status :unresolved},
          [:analysis :goal-violation-ok "a" nil]
          {:status :resolved,
           :event
           {:stream :analysis, :event :goal-violation-ok, :identifier "a"}}})
  (progress-items a)

  
  (predicate-statuses-grouped-by-predicate a)
  ;; => {[:analysis :goal-ok]
  ;;     [[[:analysis :goal-ok "a" nil]
  ;;       {:status :resolved,
  ;;        :event {:stream :analysis, :event :goal-ok, :identifier "a"}}]
  ;;      [[:analysis :goal-ok "b" nil] {:status :unresolved}]],
  ;;     [:analysis :goal-violation-ok]
  ;;     [[[:analysis :goal-violation-ok "a" nil]
  ;;       {:status :resolved,
  ;;        :event
  ;;        {:stream :analysis,
  ;;         :event :goal-violation-ok,
  ;;         :identifier "a"}}]]}

  (human-readable-progress-report
    [[[:analysis :goal-ok "a" nil]
      {:status :resolved,
       :event {:stream :analysis, :event :goal-ok, :identifier "a"}}]
     [[:analysis :goal-ok "b" nil] {:status :unresolved}]])
  ;; 
  )


(defn presentable-recommendation-status
  "Given a recommendation event and the output from
  `predicate-statuses-for-recommendation`, returns a map that can be
  used for presenting the state of the recommendation."
  [recommendation-event predicate-statuses]
  (let [recommendation (get recommendations/all (:event recommendation-event))
        resolved (keep (fn [[_k v]] (when (= (:status v) :resolved) (:event v))) predicate-statuses)
        unresolved (keep (fn [[k v]] (when (= (:status v) :unresolved) (predicate-tuple->map k))) predicate-statuses)]
    (merge
      (select-keys recommendation [:title :id :category :opportunity-paragraphs])
      {:init-time (:created recommendation-event)
       :issue-paragraphs (issue-paragraphs recommendation recommendation-event)
       :resolved resolved
       :unresolved unresolved
       :progress-percent (progress-percent (count resolved) (count unresolved))
       :progress-items (progress-items predicate-statuses)})))


(comment

  (import '(java.time Instant))
  (def rec-ev {:stream :recommendation :event :stopdcl :identifiers ["a" "b" "c"] :created (Instant/ofEpochSecond 1709740313)})
  (def pred-sts
    {[:analysis :goal-ok "a" nil] {:status :resolved :event {:stream :analysis :event :goal-ok :identifier "a" :created (Instant/ofEpochSecond 1709740313)}},
     [:analysis :goal-violation-ok "a" nil] {:status :unresolved}
     [:analysis :goal-ok "b" nil] {:status :unresolved}
     [:analysis :goal-violation-ok "b" nil] {:status :unresolved}
     [:analysis :goal-ok "c" nil] {:status :resolved :event {:stream :analysis :event :goal-ok :identifier "c" :created (Instant/ofEpochSecond 1709740313)}}
     [:analysis :goal-violation-ok "c" nil] {:status :resolved :event {:stream :analysis :event :goal-violation-ok :identifier "c" :created (Instant/ofEpochSecond 1709740313)}}})

  (presentable-recommendation-status rec-ev pred-sts )
  ;
  )


;;; TODO: this function should probably go somewhere else, since it's
;;; connecting the event extraction to the presentation
(defn conditionally-present-recommendation
  [user-id events]
  (when (not-empty events)
    (let [predicate-statuses (event-analysis/predicate-statuses-for-recommendation user-id events)]
      (presentable-recommendation-status (first events) predicate-statuses))))



