(ns codescene.features.recommendations.event-log.event-analysis
  "Extract data, draw conclusions about the state of a
  recommendation. These functions should not care which data store is
  used for the events. They should work with simple vectors of events
  that are already the result of querying the event log."
  (:require 
   [codescene.features.recommendations.definitions.recommendations :as recommendations]
   [codescene.features.recommendations.definitions.predicates :as predicates]))


(defn- as-pred-map-key
  [stream resolved identifier user-id]
  [stream resolved identifier (when (= :user stream) user-id)])

(defn- as-general-pred-map-key [stream resolved user-id]
  [stream resolved nil (when (= :user stream) user-id)])

(defn- per-item->resolution-pred-with-key
  [{:keys [stream resolved] :as _per-item} identifier user-id]
  [(as-pred-map-key stream resolved identifier user-id) {:status :unresolved}])

(defn- general-completion->resolution-pred-with-key
  [{:keys [stream resolved] :as _completion} user-id]
  [(as-general-pred-map-key stream resolved user-id) {:status :unresolved}])


(defn- resolution-predicates-from
  "Builds a map where the keys are 4-item tuples: [stream event
  identifier user-id]. user-id and identifier can be nil, depending on
  what kind of recommendation we have. user-id will only be used for
  events in the :user stream.

  The values are maps with a `:status` key that starts as
  `:unresolved`. Later, when events are processed, the status can be
  set to `:resolved` and the incoming event can be added to the
  `:event` field.

  Silently returns an emtpy map if the event type passed in is nil or
  is not a correct event."
  [user-id {:keys [event identifiers] :as _recommendation-event}]
  (let [{:keys [per-item-completion-tests completion-tests]} (get recommendations/all event {})
        ;; TODO @josf: we need to decide if this is how to handle
        ;; short-circuits. Maybe they should be a separate field on
        ;; the recommendation instead?
        per-item-completion-tests-without-short-circuit (remove :short-circuit per-item-completion-tests)
        per-item-preds (into {}
                             (mapcat
                               (fn [identifier]
                                 (map #(per-item->resolution-pred-with-key (:test %) identifier user-id) per-item-completion-tests-without-short-circuit))
                               identifiers))
        general-preds (into {}
                            (map #(general-completion->resolution-pred-with-key (:test %) user-id) completion-tests))]
    (merge per-item-preds general-preds)))

(defn remove-resolved
  [resolutions]
  (into {} (filter #(= :unresolved (:status (second %))) resolutions)))

(defn- resolution-keys
  [stream event maybe-identifier event-user-id]
  (let [current-event-is-a-resolution? (predicates/resolved-kw? event)
        current-event-unresolves? (predicates/unresolved-kw? event)]
    (when (or current-event-is-a-resolution? current-event-unresolves?)
      {:resolving-event-key (as-pred-map-key stream event maybe-identifier event-user-id)
       :unresolving-event-key (as-pred-map-key stream (predicates/resolved-keyword-for event) maybe-identifier event-user-id)
       :current-event-is-a-resolution? current-event-is-a-resolution?
       :current-event-unresolves? current-event-unresolves?})))

(defn- should-resolve?
  [resolutions current-event-is-a-resolution? resolving-event-key]
  (and current-event-is-a-resolution?
       (= :unresolved (:status (get resolutions resolving-event-key)))))

(defn- should-unresolve?
  [resolutions current-event-unresolves? unresolving-event-key]
  (and current-event-unresolves?
       (= :resolved (:status (get resolutions unresolving-event-key)))))

(defn predicate-statuses-for-recommendation
  "Given a vector of events, assumes the first item is a recommendation
  event. It then examines the subsequent events to determine which
  predicates have been resolved. Returns a map of unresolved
  predicates, where the keys are 4 item tuples: stream, event, identifier, user-id."
  [user-id events]
  (reduce
   (fn [resolutions {:keys [stream event identifier] event-user-id :user-id :as current-event}]
     (let [{:keys [current-event-is-a-resolution?
                   current-event-unresolves?
                   resolving-event-key
                   unresolving-event-key]} (resolution-keys stream event identifier event-user-id)]
       (cond
         (should-resolve? resolutions current-event-is-a-resolution? resolving-event-key)
         (assoc resolutions resolving-event-key {:status :resolved :event current-event})

         (should-unresolve? resolutions current-event-unresolves? unresolving-event-key)
         (assoc resolutions unresolving-event-key {:status :unresolved})

         :else resolutions)))
   (resolution-predicates-from user-id (first events))
   events))

