(ns codescene.features.project.pr-integration
  (:require [clojure.string :as str]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.project.core :as core]
            [codescene.features.security.tokens :as tokens]
            [medley.core :as m]
            [taoensso.timbre :as log]
            [slingshot.slingshot :refer [try+]]))

(defprotocol PrIntegrationConfig
  "PR Integration configuration snapshot for a project. It should eagerly load configuration. These instances
  should be ephemeral and not kept around for more than the scope of a web request/webhook handler,
  as they should represent config snapshot."
  (-type [this] "Type of PR Integration config")
  (-authed-client [this repo] "Authed client for this project if configured.")
  (-callback-url [this] "Callback URL, as in full CodeScene's URL plus /webhooks/azure/554 or whatever")
  (-hook-secret [this] "Hook secret")
  (-config-kv [this] "Contains config key value"))

(defmulti ci-provider (fn [pr-integration-config] (-type pr-integration-config)))

(defn hook-token-verify
  "Returns true if hook token is valid."
  [pr-integration-config {:keys [secret external-id] :as _hook}]
  (let [our-secret (-hook-secret pr-integration-config)]
    (or (nil? our-secret)
        (= our-secret secret)
        (tokens/valid-b64-salted-token? secret external-id our-secret))))

(defn our-hook-fn? [pr-integration-config]
  (fn [{:keys [callback-url]}]
    (str/starts-with? callback-url (-callback-url pr-integration-config))))

(defn- -uninstall-webhook
  "Remove webhooks from the repositories listed."
  [HookAccess PrIntegrationConfig repositories]
  (log/debugf "Uninstalling webhooks for repositories %s" (mapv :url repositories))
  (let [hooks (protocols/-repos-hooks HookAccess PrIntegrationConfig repositories)
        _ (log/debugf "Current hooks for repos %s" hooks)]
    (assert (= (count hooks) (count repositories)))
    (when-some [repos (seq (mapv merge repositories hooks))]
      (protocols/-remove-hooks HookAccess PrIntegrationConfig repos))))

(defn- user-hooks? [PrIntegrationConfig] (get-in (-config-kv PrIntegrationConfig) [:config :user-hook?]))

(defn- on-hook-check-error
  [repositories user-hooks? s]
  (map (if user-hooks?
         #(assoc % :ci-installed? true)
         #(-> % (assoc :ci-installed? false :ci-status-string s)))
       repositories))

(defrecord WebhookCIBoundary [hook-access]
  protocols/CIProvider
  (ci-installed? [this PrIntegrationConfig repositories]
    (try+
      (let [hooks (protocols/-repos-hooks hook-access PrIntegrationConfig repositories)]
        (mapv #(assoc %1
                 :ci-installed? (= :installed (:status %2))
                 :ci-status-string (:status-string %2)) repositories hooks))
      (catch [:auth-required? true] _
        (on-hook-check-error repositories (user-hooks? PrIntegrationConfig) "NOT CHECKED (credentials invalid or missing)"))
      (catch [:type :http-error] e
        (on-hook-check-error repositories (user-hooks? PrIntegrationConfig) (format "NOT CHECKED (%s)" (:message e))))))
  (enable-ci-integration! [this PrIntegrationConfig repositories]
    (when-not (user-hooks? PrIntegrationConfig)
      (log/infof "Installing webhooks for repositories %s" (mapv :url repositories))
      (let [hooks (protocols/-repos-hooks hook-access PrIntegrationConfig repositories)
            repos (mapv merge repositories hooks)
            invalid-repos (filter #(= :invalid (:status %)) repos)
            _ (when (seq invalid-repos)
                (log/warnf "%s webhook setup is in invalid state, uninstalling existing hooks, before installing."
                           (-type PrIntegrationConfig))
                (protocols/-remove-hooks hook-access PrIntegrationConfig invalid-repos))
            repos-to-install (remove #(= :installed (:status %)) repos)]
        (when (seq repos-to-install)
          (protocols/-add-hooks hook-access PrIntegrationConfig repos-to-install)))))
  (disable-ci-integration! [this PrIntegrationConfig repositories]
    (when-not (user-hooks? PrIntegrationConfig)
      (-uninstall-webhook hook-access PrIntegrationConfig repositories))))

(defn webhook-details-changed? [before after]
  (or (nil? before) (nil? after)
      (not= (-hook-secret before) (-hook-secret after))
      (not= (-callback-url before) (-callback-url after))))

(defn with-ci-integration-updates
  "This calculates the difference between two snapshots of project
  repositories and updates the CI Integration for repositories. This assumes that the repositories are the only thing
  that's changing. It is also assumed that repo :id is stable."
  [tx ProjectConfiguration action-fn error-fn]
  (let [config-before (core/-pr-integration-config ProjectConfiguration tx)
        repos-before (and config-before (core/-repositories ProjectConfiguration tx))
        ret (action-fn)
        _ (when (seq? ret) (doall ret))
        config-after (core/-pr-integration-config ProjectConfiguration tx)
        repos-after (and config-after (core/-repositories ProjectConfiguration tx))
        config (or config-after config-before)
        before (m/index-by :id repos-before)
        after (m/index-by :id repos-after)
        repos-to-disable (if (webhook-details-changed? config-before config-after)
                           repos-before
                           (remove (comp after :id) (vals before)))
        repos-to-enable (if (webhook-details-changed? config-before config-after)
                          repos-after
                          (remove (comp before :id) (vals after)))
        action (atom :disable)]
    (when-let [ci-provider (some-> config ci-provider)]
      (try
        (protocols/disable-ci-integration! ci-provider config repos-to-disable)
        (reset! action :enable)
        (protocols/enable-ci-integration! ci-provider config repos-to-enable)
        (catch Exception e
          (log/error e "Exception while enabling/disabling CI integration")
          (error-fn ret @action e))))
    ret))
