(ns codescene.features.pm-data.youtrack.youtrack-api
  "Contains methods for fetching data over the Youtrack Rest API."
  (:require [clj-http.client :as client]
            [clj-time.coerce :as tc]
            [clojure.string :as str]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [taoensso.timbre :as log]))

(defn append-api-subpath
  [params]
  (update params :api-url #(str % "/api")))

(defn- api-request
  "Core function for API requests."
  [api-client method url request-params]
  (let [params (meta-merge
                {:url url
                 :content-type :json
                 :accept       :json
                 :as           :json-strict
                 :request-method method}
                request-params)
        final-params (-> api-client
                         (api-client/augment-request params)
                         append-api-subpath ;; Note that this must be after augment-request, but before fix-api-url
                         (api-client/fix-api-url nil))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch YouTrack data from " (:url final-params))
      (client/request final-params))))

(defn get-data [api-client url query-params]
  (api-request api-client :get url
               {:query-params   query-params}))

(defn- get-paged-data
  [api-client url query-params]
  (loop [previous-data []]
    (api-client/check-interrupt)
    (log/infof "Fetching paged YouTrack data from %s, starting on %s..." url (count previous-data))
    (let [query-params (assoc query-params :$skip (count previous-data))
          {data :body} (get-data api-client url query-params)
          all-data (into previous-data data)]
      (if (seq data)
        (recur all-data)
        all-data))))

(defn- get-paged-data-using-cursors
  "This is the way paged data should be fetched according to the YouTrack API.
   The problem is that for queries with large data, it seems to be totally unpredictable and very often not return all data."
  [api-client url query-params data-key]
  (let [query-params (update query-params :fields #(str % ",hasAfter,afterCursor"))]
    (loop [cursor nil
           previous-data []]
      (api-client/check-interrupt)
      (log/infof "Fetching paged YouTrack data from %s after cursor %s (%s items fetched)..." url cursor (count previous-data))
      (let [query-params (m/assoc-some query-params :cursor cursor)
            {{:keys [hasAfter afterCursor] data data-key} :body} (get-data api-client url query-params)
            all-data (into previous-data data)]
        (if hasAfter
          (recur afterCursor all-data)
          all-data)))))

(defn- get-paged-data-using-start
  "Rather than using the intended paging mechanism, this fn uses the timestamp of the last returned item for querying for the 
   next page."
  [api-client url {:keys [start] :as query-params} data-key]
  (loop [start start
         previous-data []]
    (api-client/check-interrupt)
    (log/infof "Fetching paged YouTrack data from %s after %s (%s items fetched)..." url start (count previous-data))
    (let [query-params (m/assoc-some query-params :start start)
          {{data data-key} :body} (get-data api-client url query-params)
          all-data (into previous-data data)]
      (if (seq data)
        ;; Note that there is the unlikely possibilty that we could loose items here when incrementing
        ;; the timestamp by on. This would happen if
        ;; - more than one item have the exact same timestamp
        ;; - they happen to be on the page border (at least one item but not all of them are on the first page)
        (recur (inc (:timestamp (last data))) all-data)
        all-data))))

(defn fetch-projects [api-client]
  (get-paged-data api-client "/admin/projects" {:fields "id,name,shortName"}))

(defn fetch-custom-fields [api-client]
  (let [sub-url "/admin/customFieldSettings/customFields"
        fields "id,name,localizedName,fieldType(id),instances(id,bundle(values(name)))"]
    (get-paged-data api-client sub-url {:fields fields})))

(defn fetch-state-values [api-client state-bundle-id]
  (let [sub-url (format "/admin/customFieldSettings/bundles/state/%s/values" state-bundle-id)
        fields "id,name,localizedName,fieldType(id)"]
    (get-paged-data api-client sub-url {:fields fields})))

(defn fetch-state-bundles [api-client]
  (let [sub-url "/admin/customFieldSettings/bundles/state"
        fields "id,name,values(name,id,isResolved)"]
    (get-paged-data api-client sub-url {:fields fields})))

(defn fetch-custom-field-activities
  "Fetches all activities that are changes on custom fields"
  [api-client issue]
  (let [sub-url (format "/issues/%s/activities" issue)
        fields "timestamp,field(name),added(name)"]
    (get-paged-data api-client sub-url {:categories "CustomFieldCategory"
                                        :fields fields})))

(defn search-issues
  [api-client search-options]
  ;; Parent links seem to have direction INWARD
  (let [{:keys [since project fields]} search-options
        conditions (filter some?
                           [(when since
                              (format "updated: %s .. Today" (dates/date->string since)))
                            (when project (str "project: " project))])
        query-params {:query (str/join " " conditions)
                      :fields fields}]
    (get-paged-data api-client "/issues" query-params)))

(defn search-activities
  [api-client search-options]
  (let [{:keys [since project]} search-options
        query-params (m/assoc-some {:fields "activities(id,timestamp,field(name),added(name),target(idReadable))"
                                    :categories "CustomFieldCategory"}
                                   :issueQuery (some->> project (str "project: "))
                                   :start (some-> since tc/to-long))]
    (get-paged-data-using-start api-client "/activitiesPage" query-params :activities)))


(defn fetch-me
  [api-client]
  (let [sub-url "/users/me"]
    (:body (get-data api-client sub-url {}))))

(defn youtrack-auth
  [api-token]
  api-token)

(comment

  (def api-client (codescene.features.client.api/->ExtraProperties {:api-url "https://codescene.youtrack.cloud"} (youtrack-auth (System/getenv "YOUTRACK_TOKEN"))))
  (fetch-projects api-client)
  (->> (fetch-custom-fields api-client)
       (filter (fn [{{ :keys [id]} :fieldType}] (= "period" id))))
       ;(filter (fn [{:keys [name]}] (= "State" name))))
  (->> (fetch-state-bundles api-client ))
  (->> (search-issues api-client {:since (dates/string->date "2023-05-28")
                                  :project "DEMO"
                                  :fields "created,updated,resolved,idReadable,customFields(id,name,value(name,minutes)),links(direction,issues(idReadable))"})
       (map :idReadable))
  (->> (search-activities api-client {:since (dates/string->date "2022-05-28")
                                       :project "DEMO"}))
  (fetch-me api-client)
  )