(ns codescene.features.pm-data.youtrack.cache
  "Wraps some fetcher fns to make them cacheable"
  (:require [codescene.features.pm-data.youtrack.youtrack-fetcher :as f]
            [codescene.features.pm-data.cache :as cache]))

(def ^:private cached-data-version 1)

(defn- fetch-issues-key-fn
  [_ _ project {:keys [cost-field] :as _provider-def}]
  [project cost-field cached-data-version])

(def ^:private fetch-issues-cacheable (cache/make-cacheable f/fetch-issues ))
(cache/memo #'fetch-issues-cacheable fetch-issues-key-fn)

(defn fetch-issues*
  [since api-client project provider-def]
  (cache/combine-with-cache fetch-issues-cacheable fetch-issues-key-fn
                            since api-client project provider-def))

(defn- fetch-project-data-key-fn
  [_ _ project]
  [project cached-data-version])

(def ^:private fetch-links-cacheable (cache/make-cacheable f/fetch-links))
(cache/memo #'fetch-links-cacheable fetch-project-data-key-fn)

(defn fetch-links
  [since api-client project]
  (cache/combine-with-cache fetch-links-cacheable fetch-project-data-key-fn
                            since api-client project))

(def ^:private fetch-transitions-cacheable (cache/make-cacheable f/fetch-transitions))
(cache/memo #'fetch-transitions-cacheable fetch-project-data-key-fn)

(defn fetch-transitions
  [since api-client project]
  (cache/combine-with-cache fetch-transitions-cacheable fetch-project-data-key-fn
                            since api-client project))


(defn ->transitions-lookup
  [transitions]
  (reduce (fn [acc {:keys [issue state timestamp] :as _transition}]
            (update acc issue #(conj (or % []) [state timestamp])))
          {}
          transitions))

(defn fetch-issues
  [since api-client project provider-def]
  (let [issues (fetch-issues* since api-client project provider-def)
        ->transitions (-> (fetch-transitions since api-client project)
                          ->transitions-lookup)]
    (map (fn [{:keys [id] :as issue}]
           (assoc issue :transitions (get ->transitions id [])))
         issues)))

(comment
  (def api-client (codescene.features.client.api/->ExtraProperties {:api-url "https://codescene.youtrack.cloud"} (System/getenv "YOUTRACK_TOKEN")))
  (def project "DEMO")
  (let [since (evolutionary-metrics.trends.dates/string->date "2022-01-01")]
    (codescene.cache.core/with-file-cache "." "." :pm-data
      {:issues (count (fetch-issues since api-client project {}))
       :links (count (fetch-links since api-client project))}))
  )

