(ns codescene.features.pm-data.provider-def
  "Utility functions for parsing/unparsing provider settings."
  (:require [codescene.features.pm-data.pm-data-provider :as pm-data-provider]
            [codescene.crypto.encryptor :as encryptor]
            [clojure.string :as string]
            [medley.core :as m]))

(defn- parse-semicolon-separated [s]
  (if (seq s)
    (mapv string/trim (string/split s #";"))
    []))

(defn- parse-colon-separated [s n]
  (mapv string/trim (string/split s #":" n)))

(defn- parse-renames [s]
  (if (seq s)
    (->> s
         parse-semicolon-separated
         (mapv #(parse-colon-separated % 2))
         (filter #(= (count %) 2)))
    []))

(defn- parse-boolean [s] (some? (seq s)))

(def ^:private default-work-types #{"story" "defect" "task" "bug" "feature"})
(def ^:private default-defect-types #{"defect" "bug"})
(def ^:private default-in-progress-transitions #{"in progress" "in development" "doing"})
(def ^:private default-done-transitions #{"done" "completed" "cancelled" "rejected"})

(defn- with-default-if-empty
  "Updates the value for key with defaults taken from the available items"
  [config key default available]
  (let [value (key config)
        is-default? #(->> % :key string/lower-case (contains? default))]
    (if (empty? value)
      (assoc config key (->> available (filter is-default?) (map :key)))
      config)))


(defn- with-defaults-if-not-configured
  [current-config config-data]
  (let [{:keys [work-types transitions]} config-data
        {:keys [external-project-ids uses-supported-work-types uses-done-transitions]} current-config]
    (if (empty? external-project-ids)
      (cond-> current-config
          (and uses-supported-work-types work-types) (with-default-if-empty :supported-work-types default-work-types work-types)
          work-types (with-default-if-empty :defect-and-failure-labels default-defect-types work-types)
          transitions (with-default-if-empty :work-in-progress-transition-names default-in-progress-transitions transitions)
          (and uses-done-transitions transitions) (with-default-if-empty :work-done-transition-names default-done-transitions transitions))
      current-config)))

(defn- ->semicolon-separated-str
  [value]
  (if (coll? value) (string/join ";" value) (str value)))

(defn- ->renames-as-str
  [renames]
  (->> renames
       (map (fn [[from to]] (format "%s:%s" from to)))
       ->semicolon-separated-str))

(defn- ->boolean-str
  [value]
  (when value "on"))

(defn encrypt
  [provider-def encryptor]
  (let [encrypt (partial encryptor/encrypt encryptor)]
    (-> provider-def
        (m/update-existing :password encrypt)
        (m/update-existing :pull-requests-password encrypt))))

(defn decrypt
  [provider-def encryptor]
  (let [decrypt (partial encryptor/decrypt encryptor)]
    (-> provider-def
        (m/update-existing :password decrypt)
        (m/update-existing :pull-requests-password decrypt))))

(defn parse
  [provider-def]
  (-> provider-def
      (m/update-existing :external-project-ids parse-semicolon-separated)
      (m/update-existing :work-in-progress-transition-names parse-semicolon-separated)
      (m/update-existing :work-done-transition-names parse-semicolon-separated)
      (m/update-existing :supported-work-types parse-semicolon-separated)
      (m/update-existing :defect-and-failure-labels parse-semicolon-separated)
      (m/update-existing :rename-work-types parse-renames)
      (m/update-existing :project-aliases parse-renames)
      (m/update-existing :map-subtasks-to-parent-issues parse-boolean)
      (m/update-existing :map-commits-to-pull-request-issues parse-boolean)
      (m/update-existing :use-labels-as-work-types parse-boolean)
      ;; TODO: Having the multi-method here seems a bit complicated....
      (assoc :uses-supported-work-types (pm-data-provider/uses-supported-work-types provider-def))
      (assoc :uses-done-transitions (pm-data-provider/uses-done-transitions provider-def))))

(defn with-defaults
  [provider-def config-data]
  (-> provider-def
      (with-defaults-if-not-configured config-data)
      (update :name #(or % "Untitled"))))

(defn unparse
  [provider-def]
  (-> provider-def
      (m/update-existing :external-project-ids ->semicolon-separated-str)
      (m/update-existing :work-in-progress-transition-names ->semicolon-separated-str)
      (m/update-existing :work-done-transition-names ->semicolon-separated-str)
      (m/update-existing :supported-work-types ->semicolon-separated-str)
      (m/update-existing :defect-and-failure-labels ->semicolon-separated-str)
      (m/update-existing :rename-work-types ->renames-as-str)
      (m/update-existing :project-aliases ->renames-as-str)
      (m/update-existing :map-subtasks-to-parent-issues ->boolean-str)
      (m/update-existing :map-commits-to-pull-request-issues ->boolean-str)
      (m/update-existing :use-labels-as-work-types ->boolean-str)))
