(ns codescene.features.pm-data.jira.jira-ticket-id-extractor
  "This namespace contains fns handling ticket id extraction from commit messages."
  (:require [clojure.string :as str]
            [codescene.features.pm-data.jira.jira-cache :as jc]
            [codescene.features.pm-data.extractor-common :as ec]
            [codescene.pm-data.provider-common :as c]))

(def ticket-id-pattern #"([\w]+)-(\d+)") ;; match ids like ABC-123 (or abc-123)
 
(defn scoped-ticket-id [{:keys [scope number] :as _ticket-id}]
  (format "%s-%s" (str/upper-case scope) number))

(defn extract-ticket-ids [msg]
  (->> (re-seq ticket-id-pattern msg)
       (map (fn [[ _ scope number]] {:scope scope :number number})))) 

(defn- fetch-links [{:keys [api-client external-project-ids] :as _provider-def}]
  (->> external-project-ids
       (mapcat #(jc/fetch-links nil api-client %))))

(defn- with-resolved-parents [->parent id]
  (get ->parent id id))

(defn- make-parent-lookup [provider-def]
  (if (:map-subtasks-to-parent-issues provider-def)
    (->> (fetch-links provider-def)
         c/->parent-lookup)
    {}))

(defn make-ticket-id-extractor
  "Creates an extractor that matches ticket ids for the configured JIRA project keys."
  [{:keys [external-project-ids] :as provider-def}]
  (let [aliases (ec/aliases-from provider-def)
        ->pr-ticket-ids (ec/make-pr-ticket-ids-lookup provider-def extract-ticket-ids)
        ->parent (make-parent-lookup provider-def)]
    (fn [rev _date-time msg]
      (when (seq external-project-ids)
        (->> (concat (extract-ticket-ids msg)
                     (->pr-ticket-ids (subs rev 0 7)))
             (filter (ec/make-ticket-filter external-project-ids aliases))
             (map #(ec/with-resolved-aliases aliases %))
             (map scoped-ticket-id)
             (map #(with-resolved-parents ->parent %))))))) 

(comment
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://issues.apache.org/jira"}
                   nil))
  (def provider-def {:api-client  api-client
                     :api-url "https://issues.apache.org/jira"
                     :external-project-ids ["ZOOKEEPER"]
                     :project-aliases [["ZK" "ZOOKEEPER"]]
                     :map-subtasks-to-parent-issues true})
  (def extractor (make-ticket-id-extractor provider-def))
  (make-parent-lookup provider-def)
  (extractor "1234567890" "date not used" "zk-1411")
  )
