(ns codescene.features.pm-data.jira.jira-fetcher
  "Uses the Jira API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [taoensso.timbre :as log]
            [codescene.cache.core :as cache]
            [codescene.features.pm-data.jira.jira-api :as jira-api]
            [codescene.pm-data.provider-common :as common]
            [clojure.string :as string]
            [clojure.set :refer [rename-keys]]
            [medley.core :as m]))

(def ^:const ^:private static-jira-fields
  ["created","updated","labels","issuetype" "status"])

(defn- jira-fields-to-fetch [& dynamic-fields]
  (->> (concat static-jira-fields dynamic-fields)
       (string/join ",")))

(defn- work-types-from-labels
  [fields {:keys [supported-work-types] :as _provider-def}]
  (let [labels (set (:labels fields))]
    (->> labels
         (filterv (set supported-work-types)))))

(defn- work-types-from-issue-type
  [fields]
  (let [issue-type (get-in fields [:issuetype :name])]
    [issue-type]))

(defn- work-types-from
  [fields {:keys [use-labels-as-work-types] :as provider-def}]
  (if use-labels-as-work-types
    (work-types-from-labels fields provider-def)
    (work-types-from-issue-type fields)))

(defn- jira-seconds->minutes
  "The JIRA API reports costs in seconds, but we use minutes for our internal format.
   In the future we'll support a more dynamic configuration."
  [cost]
  (int (/ cost 60)))

(defn- cost-from
  [fields cost-field-name cost-unit]
  (let [cost (or (get fields (keyword cost-field-name)) 0)]
    (case cost-unit
      "minutes" (jira-seconds->minutes cost)
      cost)))

(defn- status-field?
  [{:keys [field]}]
  (= "status" field))

(defn- combine-time-with-status
  [{:keys [items created]}]
  (map (fn [i] (assoc i :created created)) items))

(defn- ->timestamped-transition
  [{:keys [toString created]}]
  [toString (common/drop-ms-from-time-string created)])
  
(defn- transitions-from
  [{:keys [changelog]}]
  (->> changelog
       :histories
       (mapcat combine-time-with-status)
       (filter status-field?)
       (map ->timestamped-transition)
         ;; The dates are from new to old -> reverse
       reverse
       common/keep-first-transitions
       (into [])))

(defn- issue->response
  [provider-def issue]
  (let [{:keys [cost-field  cost-unit]} provider-def
        {:keys [fields key]} issue]
    {:id key
     :created  (common/drop-ms-from-time-string (:created fields))
     :closed nil ;; Jira issues have no closed property!
     :updated (:updated fields)
     :status (get-in fields [:status :name])
     :cost (cost-from fields cost-field cost-unit)
     :work-types (work-types-from fields provider-def)
     :transitions (transitions-from issue)}))

(defn- subtask->response 
  "Returns nil if the subtask was not valid (ie had no parent link)"
  [issue]
  (let [{:keys [fields key]} issue
        parent (get-in fields [:parent :key])]
    (when (some? parent)
      {:id key
       :parent parent
       :updated (:updated fields)})))

(defn- item-name
  [{:keys [untranslatedName name] :as _item}]
  ;; CS-21245: Note that the toString in changelog statuses for Jira cloud matches untranslatedName, not name.
  (or untranslatedName name))

(defn- fetch-issue-types-in-project
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client key]
  (log/info "Fetch issue types")
  (->> (jira-api/fetch-issue-types-in-project api-client key)
       (mapv item-name)))

(defn fetch-issues
  [since api-client project provider-def]
  (let [{:keys [cost-field pm-data-page-size]} provider-def
        options (m/assoc-some {} :page-size pm-data-page-size)
        search-options {:projects [project]
                        :since since}
        result-options {:fields (jira-fields-to-fetch cost-field)
                        :changelog true
                        :transformer (partial issue->response provider-def)}]
    (log/infof "Fetch issues from Jira since %s" (or since "-"))
    (->> (jira-api/search api-client options search-options result-options)
         (into []))))

(defn fetch-issue-types
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (log/info "Fetch issue types")
  (->> (jira-api/fetch-issue-types api-client)
       (map item-name)
       distinct
       (mapv common/->name-and-key)))

(defn- fetch-subtask-types
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (->> (jira-api/fetch-issue-types api-client)
       (filter :subtask)
       (map item-name)
       (into [] (distinct))))

(defn fetch-number-fields
  "Fetches fields from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (log/info "Fetch fields")
  (->> (jira-api/fetch-fields api-client)
       (filter #(= "number" (get-in % [:schema :type])))
       ;; Note: In Jira server key is not present in reply -> use id and rename
       (map #(select-keys % [:name :id]))
       (mapv #(rename-keys % {:id :key}))))

(defn fetch-labels
  "Fetches labels from the remote JIRA API. Throws when the API calls fail."
  [api-client api-url]
  (log/info "Fetch labels")
  (if (jira-api/is-jira-cloud? api-url)
    (->> (jira-api/fetch-labels api-client)
         (mapv common/->name-and-key))
    ;; There's no endpoint for fetching labels from Jira Server... see https://jira.atlassian.com/browse/JRASERVER-29409
    ;; (the user will have to specify the labels to use as work types manually in the uu)
    []))

(defn fetch-transitions
  "Fetches transitions from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (log/info "Fetch transitions")
  (->> (jira-api/fetch-statuses api-client)
       ;; Note that statuses (can) have different ids in different projects, thus using id as key is not really doable with the current solution.
       (map item-name)
       distinct
       (mapv common/->name-and-key)))

(defn fetch-links
  [since api-client project]
  (let [search-options {:projects [project]
                        :issue-types (fetch-subtask-types api-client)
                        :since since}
        result-options {:fields ["parent" "updated"]
                        :transformer subtask->response}]
    (log/infof "Fetch links from Jira since %s" (or since "-"))
    (->> (jira-api/search api-client {} search-options result-options)
         (filterv some?))))

(defn fetch-projects
  "Fetches projects from the remote JIRA API. Throws when the API calls fail."
  [api-client api-url]
  (->> (if (jira-api/is-jira-cloud? api-url)
         (jira-api/fetch-projects-for-cloud api-client)
         (jira-api/fetch-projects-for-server api-client))
       (mapv #(select-keys % [:name :key]))))

(def fetch-myself jira-api/fetch-myself)

(cache/memo-scoped #'fetch-issue-types)
(cache/memo-scoped #'fetch-subtask-types)
(cache/memo-scoped #'fetch-labels)
(cache/memo-scoped #'fetch-transitions)
(cache/memo-scoped #'fetch-projects)

(comment

  (def api-url "https://empear.atlassian.net")
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url api-url}
                   [(System/getenv "JIRA_USER") (System/getenv "JIRA_PASSWORD")]))
  (def project "ENTERPRISE")
  (def api-url "https://issues.apache.org/jira")
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url api-url}
                   nil))
  (def project "ZOOKEEPER")
  (def provider-def {:supported-work-types ["Task" "Story"]
                     :pm-data-page-size 25
                     :cost-field "timeoriginalestimate"
                     :cost-unit "minutes"})
  (def since (evolutionary-metrics.trends.dates/string->date "2020-02-25"))
  (fetch-labels api-client api-url)
  (fetch-transitions api-client)
  (fetch-issue-types api-client)
  (fetch-subtask-types api-client)
  (fetch-issue-types-in-project api-client project)
  (fetch-number-fields api-client)
  (fetch-projects api-client api-url)
  (fetch-issues since api-client project provider-def)
  (fetch-links since api-client project)
  )
