
(ns codescene.features.pm-data.jira.jira-api
  "Contains methods for fetching data over the Jira Rest API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require [clj-http.client :as client]
            [taoensso.timbre :as log]
            [clojure.string :as string]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]))

(def ^:const ^:private jira-cloud-rest-api-path
  "/rest/api/3")

(def ^:const ^:private jira-server-rest-api-path
  "/rest/api/2")

(defn is-jira-cloud? [api-url]
  (string/includes? api-url "atlassian.net"))

(defn api-subpath [api-url]
  (if (is-jira-cloud? api-url)
    jira-cloud-rest-api-path
    jira-server-rest-api-path))

(defn append-api-subpath
  [params] 
  (update params :api-url #(str % (api-subpath %))))

(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :content-type :json
                  :accept       :json
                  :as           :json-strict
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         append-api-subpath ;; Note that this must be after augment-request, but before fix-api-url
                         (api-client/fix-api-url nil))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch Jira data from " (:url final-params))
      (client/request final-params))))

(defn api-request
  [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(def ^:const default-http-page-size 100)

(defn get-data [api-client url query-params]
  (api-request api-client :get url
              {:query-params   query-params}))

(defn- get-paged-data 
  [api-client 
   url 
   {:keys [data-field query-params page-size transformer]
    :or {query-params {} page-size default-http-page-size transformer identity} :as options}]
  (loop [start-at 0
         max-results (or page-size default-http-page-size)
         total-count nil
         acc-data []]
    (api-client/check-interrupt)
    (log/infof "Fetching paged jira data from %s starting at %d%s..." 
               url start-at (if total-count (format " (of %s)" total-count) "") )
    (let [query-params (merge query-params
                              {:startAt start-at
                               :maxResults max-results})
          result (get-data api-client url query-params)
          {max-results-received :maxResults data data-field :keys [total isLast]} result]
      (if (or isLast (<= total (+ (count data) (count acc-data))))
        (into acc-data (map transformer) data)
        (recur (+ max-results-received start-at) max-results-received total (into acc-data (map transformer) data))))))

(defn fetch-project
  "Fetches project from the remote JIRA API. Throws when the API calls fail."
  [api-client project]
  (let [url (format "/project/%s" project)]
    (get-data api-client url {})))

(defn fetch-issue-types-in-project
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client project]
  (let [url (format "/project/%s" project)]
    (get-data api-client url {})))

(defn fetch-issue-types
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/issuetype" {}))

(defn fetch-fields
  "Fetches fields from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/field" {}))

(defn fetch-labels
  "Fetches labels from the remote JIRA API. Throws when the API calls fail.
(Jira Server has no API for getting a list of labels?!)"
  [api-client]
  (get-paged-data api-client  "/label" {:data-field :values}))

(defn fetch-statuses
  "Fetches statuses from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/status" {}))

(defn- join-double-quoted
  [separator labels]
  (string/join separator (map #(format "\"%s\"" %) labels)))

(defn fetch-projects-for-cloud [api-client]
  (get-paged-data api-client "/project/search" {:data-field :values
                                                :query-params {:jql ""}}))

(defn fetch-projects-for-server [api-client]
  (get-data api-client "/project" {:jql ""}))

(defn- in-condition [name coll]
  (format "%s in (%s)" name (join-double-quoted  "," coll)))

(defn- updated-condition [since]
  ;; Jira apparently only supports dates in jql filters 
  (format "updated> \"%s \"" (dates/date->string since)))

(defn- not-empty-coll [coll]
  (not (and (coll? coll) (empty? coll))))

(defn search
  "Fetches issues from the remote JIRA API. Returns nil when the API calls fail."
  [api-client options search-options result-options]
  (let [{:keys [page-size]} options
        {:keys [projects issue-types labels issues since]} search-options
        {:keys [fields changelog transformer] :or {transformer identity}} result-options
        conditions (filter some?
                           [(when (seq projects) (in-condition "project" projects))
                            (when (seq issue-types) (in-condition "issuetype" issue-types))
                            (when (seq labels) (in-condition "labels" labels))
                            (when (seq issues) (in-condition "key" issues))
                            (when (some? since) (updated-condition since))])
        jql (str (string/join " AND " conditions)
                 " order by updated desc")]
    ;; JIRA doesn't allow querys like 'projects in ()' -> filter here
    (if (and (not-empty-coll projects)
             (not-empty-coll issue-types)
             (not-empty-coll issues)
             (not-empty-coll labels))
      (get-paged-data api-client "/search"
                      {:data-field :issues
                       :page-size page-size
                       :query-params {:jql jql
                                      :fields fields
                                      :expand (when changelog "changelog")}
                       :transformer transformer})
      
      [])))

(defn fetch-myself
  "Fetches info about the authenticated user from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/myself" {}))

(defn jira-auth
  [username password]
  (cond
    (and (seq username) (seq password)) [username password]
    (seq password) password
    :else nil))

(comment
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://empear.atlassian.net"}
                   (jira-auth (System/getenv "JIRA_USER") (System/getenv "JIRA_PASSWORD"))))
  (def project "ENTERPRISE")
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://issues.apache.org/jira"}
                   (jira-auth "" "")))
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "http://localhost:8080"}
                   (jira-auth "" (System/getenv "JIRA_SERVER_TOKEN"))))
  (def project "ZOOKEEPER")
  (map (comp :updated :fields)
       (search api-client
               {:page-size 25}
               {:projects [project]
                     ;:issue-types ["Story"]
                :since (dates/string->date "2020-01-26")
                :labels nil}
               {:fields ["created" "updated" "labels" "issuetype" "status"]
                :changelog true}))
  (fetch-project api-client project)
  (fetch-issue-types-in-project api-client project)
  (fetch-issue-types api-client)
  (fetch-fields api-client)
  (fetch-labels api-client)
  (fetch-statuses api-client)
  (map :name (fetch-projects-for-cloud api-client))
  (map :name (fetch-projects-for-server api-client))
  (fetch-myself api-client))