(ns codescene.features.pm-data.gitlab.gitlab-provider
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider]]
            [codescene.features.pm-data.gitlab.cache :as cache]
            [codescene.features.pm-data.gitlab.fetcher :as f]
            [codescene.features.pm-data.gitlab.gitlab-ticket-id-extractor :as x]
            [codescene.pm-data.provider-common :as common]
            [taoensso.timbre :as log]))

(defn- get-pm-data
  [provider-def {:keys [repo-urls since] :as _pm-data-context}]
  (let [{:keys [api-client]} provider-def
        issues (->> repo-urls
                    (mapcat #(cache/fetch-issues since api-client % provider-def))
                    (into []))]
    {:issues issues
     :provider-def provider-def}))

(defn- get-configuration-data [provider-def {:keys [repo-urls] :as _pm-data-context}]
  (log/info "Fetching configuration data from GitLab")
  (let [labels (f/fetch-labels (:api-client provider-def) repo-urls)
        board-labels (f/fetch-board-labels (:api-client provider-def) repo-urls)]
    {:work-types labels
     :projects nil
     :transitions (concat [(common/->name-and-key "Opened")]
                          board-labels
                          [(common/->name-and-key "Closed")])}))

(defn- validate-settings
  [{:keys [api-client] :as _provider-def} {:keys [repo-urls] :as _pm-data-context}]
  (f/fetch-labels api-client (take 1 repo-urls))
  nil)

(defrecord GitLabProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this repo] (x/make-ticket-id-extractor provider-def repo))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def pm-data-context))
  (-validate-settings [_this] (validate-settings provider-def pm-data-context)))

(defmethod -create-pm-data-provider "gitlab"
  [provider-def pm-data-context]
  (->GitLabProvider provider-def pm-data-context))

(comment
  (def repo-url "https:empear/analysis-target.git")
  (def since (evolutionary-metrics.trends.dates/string->date "2021-07-04"))
  (def context {:repo-urls [repo-url]
                :since since})
  (def provider-def {:rename-work-types []
                     :defect-and-failure-labels []
                     :work-in-progress-transition-names ["In Progress"]
                     :work-done-transition-names ["Done"]
                     :api-client (System/getenv "GITLAB_TOKEN")})
  (get-configuration-data provider-def context)
  (validate-settings provider-def context)
  (get-pm-data provider-def context))