(ns codescene.features.pm-data.github.github-fetcher
  "Uses the GiHub API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [taoensso.timbre :as log]
            [codescene.cache.core :as cache]
            [codescene.features.pm-data.github.github-api :as github-api]
            [codescene.features.pm-data.github.github-api-v4 :as github-api-v4]
            [codescene.pm-data.provider-common :as common]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- work-types-from
  [labels supported-work-types]
  (let [label-names (map :name labels)]
    (filterv (set supported-work-types) label-names)))

(defn fetch-labels
  "Fetches labels from the remote GitHub API. Throws when the API calls fail."
  [api-client repos]
  (->> repos
       (mapcat (fn [{:keys [owner repo]}]
                 (github-api-v4/fetch-labels api-client owner repo)))
       (map :name)
       distinct
       (mapv common/->name-and-key)))

(defn- pr->result [{:keys [number title headRefName bodyText commits mergeCommit updatedAt] :as _pr}]
  {:id (str number)
   :title title
   :branch headRefName
   :text bodyText
   :merge-commit (:oid mergeCommit)
   :commits (->> commits :nodes
                 (mapv (comp :oid :commit)))
   :updated (dates/timestamp->date updatedAt)})

(defn- pr->pr-number [{:keys [number updatedAt] :as _pr}]
  {:id number
   :updated (dates/timestamp->date updatedAt)})

(defn fetch-pull-requests
  "Fetches pull-requests from the remote GitHub API. Throws when the API calls fail."
  [since api-client owner repo]
  (log/infof "Fetch pull requests from GitHub since %s" (or since "-"))
  (let [search-options {:since since}]
    (->> (github-api-v4/fetch-pull-requests api-client owner repo search-options)
         (mapv pr->result))))

(defn fetch-pull-request-count
  "Fetches pull-requests count from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo]
  (github-api-v4/fetch-pull-request-count api-client owner repo))

(defn fetch-pull-request-numbers
  "Fetches pull-requests from the remote GitHub API. Throws when the API calls fail."
  [since api-client owner repo]
  (log/infof "Fetch pull request numbers from GitHub since %s" (or since "-"))
  (let [search-options {:since since}]
    (->> (github-api-v4/fetch-pull-request-numbers api-client owner repo search-options)
         (mapv pr->pr-number))))

(defn fetch-user-type [api-client owner]
  (->> (github-api/fetch-user api-client owner)
       :type))

(defn- issue->result
  [provider-def issue]
  (let [{:keys [number createdAt closedAt updatedAt labels]} issue
        labels (->> labels :nodes)
        {:keys [supported-work-types]} provider-def]
    {:id (str number)
     :created  (common/drop-ms-from-time-string createdAt)
     :closed  (common/drop-ms-from-time-string closedAt)
     :updated updatedAt
     :cost 0
     :status nil
     :transitions []
     :work-types (work-types-from labels (set supported-work-types))}))

(defn fetch-issues
  "Fetches issues from the remote GitHub API. Throws when the API calls fail."
  [since api-client owner repo provider-def]
  (log/infof "Fetch issues from GitHub since %s" (or since "-"))
  (let [search-options {:since since}]
    (->> (github-api-v4/fetch-issues api-client owner repo search-options)
         (mapv #(issue->result provider-def %)))))
 
(cache/memo-scoped #'fetch-labels)

(comment
  (def api-client (System/getenv "GITHUB_TOKEN"))
  (def owner "dotnet")
  (def repo "aspnetcore")
  (def provider-def {:supported-work-types ["feature-razor.vscode"]})
  ;(fetch-labels api-client owner repo)
  (fetch-pull-requests "2022-01-22T22:23:21Z" api-client owner repo)
  (fetch-pull-request-count api-client owner repo)
  (fetch-user-type api-client owner)
  (fetch-issues (evolutionary-metrics.trends.dates/string->date "2025-04-01") api-client owner repo provider-def)
  )
