(ns codescene.features.pm-data.github.github-api-v4
  "Contains methods for fetching data over the GitHub Graph QL API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require [taoensso.timbre :as log]
            [slingshot.slingshot :refer [throw+]]
            [clj-time.core :as tc]
            [clojure.string :as string]
            [codescene.features.client.api :refer [check-interrupt]]
            [codescene.features.repository-provider.github.api :as github-api]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- join-double-quoted
  [separator labels]
  (string/join separator (map #(format "\"%s\"" %) labels)))

(defn- get-data
  [api-client query]
  (h/with-http-error-messages (format "Failed to fetch GitHub data using query: %s" query)
    (github-api/graphql-request api-client query)))

(defn- continue? [data]
  (-> data :pageInfo :hasNextPage true?))

(defn- get-paged-data
  "Does a paged graphQL query"
  ([api-client query-fn data-fn]
   (get-paged-data api-client query-fn data-fn continue?))
  ([api-client query-fn data-fn continue-fn]
   (loop [after nil
          totalCount nil
          previous-nodes []]
     (check-interrupt)
     (log/infof "Fetching paged github data after cursor %s %s..." after
                (if after (format "(%s of %s)" (count previous-nodes) totalCount) ""))
     (let [paging-str (str "first: 100" (when after (format ",after:\"%s\"" after)))
           query (query-fn paging-str)
           {:keys [body]} (get-data api-client query)
           error (-> body :errors first :message)
           _ (when error (throw+ {:type :http-error
                                  :message (format "Failed to fetch GitHub data (%s) using query: %s" error query)}))
           data (data-fn body)
           {:keys [pageInfo totalCount issueCount]} data
           {:keys [endCursor]} pageInfo
           all-nodes (concat previous-nodes (:nodes data))]
       (if (continue-fn data)
         (recur endCursor (or totalCount issueCount) all-nodes)
         all-nodes)))))

(defn- make-issues-query-fn
  "Creates a fn that takes the paging part as input and creates an issues query"
  [owner repo filters order-by]
  (fn [paging]
    (format
     "query {
      repository(owner:\"%s\", name:\"%s\") {
        issues(%s, filterBy: {%s}, orderBy: %s) {
          pageInfo {
            startCursor
            hasNextPage
            endCursor
          }
          totalCount
        
          nodes {
            number
            createdAt
            closedAt
            updatedAt
            labels(first:100) {
              nodes {
                name
              }
            }
          }
        }
      }
    }" owner repo paging (string/join "," filters) order-by)))

(defn- make-labels-query-fn
  "Creates a fn that takes the paging part as input and creates an projects query"
  [owner repo]
  (fn [paging]
    (format
     "query {
      repository(owner:\"%s\", name:\"%s\") {
        labels(%s) {
          pageInfo {
            startCursor
            hasNextPage
            endCursor
          }
          totalCount
          nodes {
            name
          }
        }
      }
    }" owner repo paging)))



(defn- make-pull-requests-query-fn
  "Creates a fn that takes the paging part as input and creates an pull requests query.
  Use 50-item pages (instead of 100) to try to workaround occasional timeouts - see https://app.clickup.com/t/9015696197/CS-4251"
  [owner repo order-by]
  (fn [paging]
    (format
     "query {
    repository(owner: \"%s\", name: \"%s\"){
      pullRequests(%s, orderBy: %s){
        pageInfo {
          startCursor
          hasNextPage
          endCursor
        }
        totalCount
        nodes{
          number
          title
          bodyText
          headRefName
          updatedAt
          mergeCommit {oid}
          commits(first: 50) {
            nodes {
              commit {
                oid
              }
            }
          }
        }
      }
    }
    }" owner repo paging order-by)))

(defn- make-pull-requests-numbers-query-fn
  "Creates a fn that takes the paging part as input and creates an pull requests query"
  [owner repo order-by]
  (fn [paging]
    (format
     "query {
    repository(owner: \"%s\", name: \"%s\"){
      pullRequests(%s, orderBy: %s){
        pageInfo {
          startCursor
          hasNextPage
          endCursor
        }
        totalCount
        nodes{
          number
          updatedAt
        }
      }
    }
    }" owner repo paging order-by)))

(defn- make-pull-requests-count-query
  "Creates a pull requests count query"
  [owner repo]
  (format
   "query {
    repository(owner: \"%s\", name: \"%s\"){
      pullRequests(first: 0){
        totalCount
      }
    }
    }" owner repo))

(defn- make-specific-issues-query-fn
  "Creates a fn that takes the paging part as input and creates an issues query.
   Actually searches for the ticked ids in the default fields, meaning that it will
   return issues having the nbrs in titles/comments and not just the issues with those nbrs"
  [owner repo ticket-ids]
  (fn [paging]
    (format
     "query {
    search(query: \"repo:%s/%s is:issue %s\", type: ISSUE, %s){
      pageInfo {
        startCursor
        hasNextPage
        endCursor
      }
      issueCount
      nodes {
        ... on Issue {
          number
          createdAt
          closedAt
          labels(first:100) {
            nodes {
              name
            }
          }
        }
      }
    }
      }" owner repo (string/join " " ticket-ids)
       paging)))

(def order-by-update-at-desc "{field: UPDATED_AT, direction: DESC}")

(defn fetch-specific-issues
  "Fetches issues from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo ticket-ids]
  (let [query-fn (make-specific-issues-query-fn owner repo ticket-ids)
        data-fn (comp :search :data)]
    (get-paged-data api-client query-fn data-fn)))

(defn fetch-issues
  "Fetches issues from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo search-options]
  (let [{:keys [labels since]} search-options
        filters (filter some?
                  [(when (seq labels) (format "labels:[%s]" (join-double-quoted " " labels)))
                   (when since (format "since:\"%s\"" (dates/date-time->string since)))])
        query-fn (make-issues-query-fn owner repo filters order-by-update-at-desc)
        data-fn (comp :issues :repository :data)]
    (get-paged-data api-client query-fn data-fn)))

(defn fetch-labels
  "Fetches labels from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo]
  (let [query-fn (make-labels-query-fn owner repo)
        data-fn (comp :labels :repository :data)]
    (get-paged-data api-client query-fn data-fn)))

(defn- ->not-old-enough-fn [since]
  (fn [data]
    (and (continue? data)
         (when-let [updated-at (some-> data :nodes last :updatedAt dates/date-time-string->date)]
           (tc/before? since updated-at)))))

(defn fetch-pull-requests
  "Fetches pull requests from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo search-options]
  (let [{:keys [since]} search-options
        query-fn (make-pull-requests-query-fn owner repo order-by-update-at-desc)
        data-fn (comp :pullRequests :repository :data)
        ;; Since there is no API for filtering pullrequests by update date
        ;; we have to manage it ourselves...
        continue-fn (if since 
                      (->not-old-enough-fn since)
                      continue?)]
    (get-paged-data api-client query-fn data-fn continue-fn)))

(defn fetch-pull-request-numbers
  "Fetches pull requests from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo search-options]
  (let [{:keys [since]} search-options
        query-fn (make-pull-requests-numbers-query-fn owner repo order-by-update-at-desc)
        data-fn (comp :pullRequests :repository :data)
        continue-fn (if since
                      (->not-old-enough-fn since)
                      continue?)]
    (get-paged-data api-client query-fn data-fn continue-fn)))

(defn fetch-pull-request-count
  "Fetches pull requests count from the remote GitHub API. Throws when the API calls fail."
  [api-client owner repo]
  (let [query (make-pull-requests-count-query owner repo)
        data-fn (comp :totalCount :pullRequests :repository :data :body)]
    (-> (get-data api-client query)
        data-fn)))

(comment
  (def api-token (System/getenv "GITHUB_TOKEN"))
  (def owner "dotnet")
  (def owner "knorrest")
  (def repo "aspnetcore")
  (def repo "analysis-target")

  (println ((make-issues-query-fn owner repo nil order-by-update-at-desc) "first: 1"))
  (println ((make-specific-issues-query-fn owner repo [1 2 8]) "first: 1"))
  (println ((make-pull-requests-query-fn owner repo order-by-update-at-desc) "first: 1"))
  (fetch-specific-issues api-token owner repo [39201])
  (fetch-issues api-token owner repo {} #_{:since (dates/date-time-string->date "2022-01-22T22:23:21Z")})
  (fetch-labels api-token owner repo)
  (fetch-pull-requests api-token owner repo {:since (dates/date-time-string->date "2022-01-22T22:23:21Z")})
  (fetch-pull-request-numbers api-token owner repo {:since (dates/date-time-string->date "2022-01-22T22:23:21Z")})
  (fetch-pull-request-count api-token owner repo)
  )
