
(ns codescene.features.pm-data.clickup.api
  (:require [clj-http.client :as client]
            [clj-time.coerce :as coerce]
            [taoensso.timbre :as log]
            [clojure.string :as string]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]))

(def api-url "https://api.clickup.com/api/v2")

(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                {:url url
                 :content-type :json
                 :accept       :json
                 :as           :json-strict
                 :request-method method}
                request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url api-url))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch ClickUp data from " (:url final-params))
      (client/request final-params))))

(defn api-request
  [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(def ^:const default-http-page-size 100)

(defn get-data [api-client url query-params]
  (api-request api-client :get url
               {:query-params   query-params}))

(defn- get-paged-data [api-client url data-field query-params]
  (loop [page 0
         acc-data []]
    (api-client/check-interrupt)
    (log/infof "Fetching page %s of ClickUp data from %s..." page url)
    (let [query-params (merge query-params
                              {:page page})
          {data data-field :keys [last_page]} (get-data api-client url query-params)]
      (if (not (false? last_page))
        (into acc-data data)
        (recur (inc page) (into acc-data data))))))

(defn fetch-teams
  [api-client]
  (-> (get-data api-client "/team" {})
      :teams))

(defn fetch-team
  [api-client team-id]
  (let [url (format "/team/%s" team-id)]
    (get-data api-client url {})))

(defn fetch-spaces
  [api-client team-id]
  (let [url (format "/team/%s/space" team-id)]
    (-> (get-data api-client url {:archived false})
        :spaces)))

(defn fetch-space
  [api-client space-id]
  (let [url (format "/space/%s" space-id)]
    (get-data api-client url {})))

(defn fetch-folders
  [api-client space-id]
  (let [url (format "/space/%s/folder" space-id)]
    (-> (get-data api-client url {:archived false})
        :folders)))

(defn fetch-folder
  [api-client folder-id]
  (let [url (format "folder/%s" folder-id)]
    (get-data api-client url {})))

(defn fetch-lists
  [api-client folder-id]
  (let [url (format "folder/%s/list" folder-id)]
    (-> (get-data api-client url {:archived false})
        :lists)))

(defn fetch-list
  [api-client list-id]
  (let [url (format "list/%s" list-id)]
    (get-data api-client url {})))

(defn fetch-folderless-lists
  [api-client space-id]
  (let [url (format "/space/%s/list" space-id)]
    (-> (get-data api-client url {:archived false})
        :lists)))

(defn fetch-custom-task-types
  [api-client team-id]
  (let [url (format "/team/%s/custom_item" team-id)]
    (-> (get-data api-client url {})
        :custom_items)))

(defn- make-it-at-least-two-items
  "Arrrghh"
  [[first & rest :as items]]
  (when first
    (cond-> items
      (empty? rest) (conj first))))

(defn fetch-filtered-team-tasks
  [api-client team-id {:keys [since spaces lists] :as _search-options}]
  (let [url (format "/team/%s/task" team-id)
        query-params (m/assoc-some {:order_by "updated"
                                    :subtasks true
                                    :include_closed true
                                    :reverse false}
                                   :space_ids (make-it-at-least-two-items spaces)
                                   :list_ids (make-it-at-least-two-items lists)
                                   :date_updated_gt (when since (coerce/to-long since)))]
    (get-paged-data api-client url :tasks query-params)))

(defn- fetch-bulk-tasks-time-in-status*
  [api-client task-ids]
  (let [url "/task/bulk_time_in_status/task_ids"
        query-params {:task_ids task-ids}]
    (get-data api-client url query-params)))

(defn fetch-bulk-tasks-time-in-status
  "Since the endpoint for this supports querying for 100 issues at a time,
   partition the task-ids and do several calls as necessary"
  [api-client task-ids]
  (reduce (fn [acc ids]
            (merge acc 
                   (->> (fetch-bulk-tasks-time-in-status* api-client ids)
                        (m/map-keys name)))) ;; we don't want ids to be keywords
          {}
          (partition-all 100 task-ids)))

(defn clickup-auth
  [api-token]
  (reify api-client/ApiClient
    (augment-request [_this req] (assoc-in req [:headers "Authorization"] api-token))
    (on-expired-token [_this] nil)))

(comment

  (def api-client (clickup-auth (System/getenv "CLICKUP_TOKEN")))
  (def team-name "CodeScene AB")
  (def team  (->> (fetch-teams api-client)
                  (m/find-first #(= team-name (:name %)))
                  :id))
  (fetch-teams api-client)
  (fetch-team api-client team)

  (->> (fetch-teams api-client)
       (map :name))
  (->> team
       (fetch-spaces api-client)
       (map :name))
  (->> team
       (fetch-custom-task-types api-client)
       (map :name))
  (->> team
       (fetch-spaces api-client) first :id
       (fetch-folders api-client)
       (map :name))


  (->> team
       (fetch-spaces api-client) first :id
       (fetch-space api-client))

  (->> team
       (fetch-spaces api-client) first :id
       (fetch-folders api-client) first :id
       (fetch-lists api-client)
       (map :name))

  (->> team
       (fetch-spaces api-client) first :id
       (fetch-folderless-lists api-client)
       (map :name))

  (->> team
       (fetch-spaces api-client) first :id
       (fetch-folders api-client) first :id
       (fetch-lists api-client) first :id
       (fetch-list api-client)
       :statuses (map :status))

  (def tasks (let [spaces (->> (fetch-spaces api-client team) (map :id))]
               (fetch-filtered-team-tasks api-client team {:spaces spaces})))
  (def tasks (fetch-filtered-team-tasks api-client team {:lists ["901505265095"]
                                                         :since "2024-07-12"}))
  (count tasks)

  (fetch-bulk-tasks-time-in-status api-client (map :id tasks))
  )