(ns codescene.features.pm-data.azure.azure-cache
  "Wraps some fetcher fns to make them cacheable"
  (:require [codescene.features.pm-data.azure.azure-fetcher :as f]
            [codescene.features.pm-data.cache :as cache]
            [medley.core :as m]))

(defn fetch-pull-request-key-fn
  [_ {:keys [owner project] :as _repo-parts} pr {:keys [api-url] :as _provider-def}]
  [api-url owner project pr])

(defn- fetch-pull-request
  [api_client repo-parts pr _provider-def]
  (f/fetch-pull-request api_client repo-parts pr))

(cache/memo #'fetch-pull-request fetch-pull-request-key-fn)

(defn fetch-pull-requests
  [api-client repo-parts provider-def]
  (->> (f/fetch-pull-requests api-client repo-parts)
       (mapv #(fetch-pull-request api-client repo-parts % provider-def))))

(defn fetch-work-item-revisions-key-fn
  [_ _ {:keys [owner project] :as _repo-parts} {:keys [api-url cost-field] :as _provider-def}]
  [api-url owner project cost-field])

(def ^:private fetch-work-item-revisions-cacheable (cache/make-cacheable f/fetch-work-item-revisions))
(cache/memo #'fetch-work-item-revisions-cacheable fetch-work-item-revisions-key-fn)
  
(defn fetch-issues
  [since api-client repo-parts provider-def]
  (->> (cache/combine-with-cache fetch-work-item-revisions-cacheable fetch-work-item-revisions-key-fn
                                 since api-client repo-parts provider-def)
       f/group-by-work-item
       (map (partial f/revisions->issue provider-def))))

(defn- fetch-links-key-fn
  [_ _ {:keys [owner project] :as _repo-parts} {:keys [api-url] :as _provider-def}]
  [api-url owner project])

(def ^:private fetch-links-cacheable (cache/make-cacheable f/fetch-links))
(cache/memo #'fetch-links-cacheable fetch-links-key-fn)

(defn fetch-links
  [since api-client repo-parts provider-def]
  (cache/combine-with-cache fetch-links-cacheable fetch-links-key-fn 
                            since api-client repo-parts provider-def))

(comment
  (def api-client ["" (System/getenv "AZURE_TOKEN")])
  (def repo-url "git@ssh.dev.azure.com:v3/empear/SmartHotel360/PublicWeb")
  (let [since (evolutionary-metrics.trends.dates/string->date "2021-01-14")
        repo-parts (codescene.url.url-utils/azure-url->parts repo-url)]
    (codescene.cache.core/with-file-cache "." "." :pm-data
      (medley.core/find-first #(= "48" (:id %)) (fetch-work-item-revisions since api-client repo-parts {}))))
  )
