(ns codescene.features.plugins.project-configuration-extension
  "Wraps an external ProjectConfigurationExtension in an implementation of 
  the internal ConfigurationExtension protocol."
  (:require [codescene.features.plugins.configuration :as c]
            [codescene.features.plugins.configuration-extension :refer [ConfigurationExtension] :as _ce]
            [codescene.features.plugins.settings-db :as db]
            [clj-codescene-plugin.setting :as setting]
            [clj-codescene-plugin.configuration :as configuration]
            [taoensso.timbre :as log])
  (:import [com.codescene.plugin.configuration ValidationException]))
 
(defn ->Context
  "Provides a Context instance for the external ProjectConfigurationExtension"
  [encryptor global-settings]
  (configuration/->Context {:global-settings (c/decrypt-values encryptor global-settings)}))

(defn- settings [extension encryptor global-settings]
  (map setting/Setting->map (.getSettings extension (->Context encryptor global-settings))))

(defn- validate-settings [extension encryptor global-settings settings]
  (.validateSettings extension
                     (->Context encryptor global-settings) 
                     (map setting/->SettingValue (c/decrypt-values encryptor settings))))

(defn- stored-or-default-settings
  "Note that stored values for encrypted setting types will still be encrypted.
   If no values are stored, the default values from the extension will be used"
  [plugin-id settings {:keys [db] :as _context}]
  (let [stored-values (db/settings-for db plugin-id)]
    (c/synchronize-settings settings stored-values)))

(defn- store-settings
  [plugin-id name extension {:keys [global-settings db encryptor] :as _context} values]
  (try
    (let [settings (c/synchronize-settings (settings extension encryptor global-settings) values)]
      (validate-settings extension encryptor global-settings settings)
      (let [storeable-settings (c/encrypt-values encryptor settings)]
        (db/update-settings db plugin-id name storeable-settings)))
    (catch ValidationException e
      (.details e))
    (catch Exception e
      (let [msg (format "Failed to do validation in extension %s" name)]
        (log/warnf e msg)
        [msg]))))

(defn ->ConfigurationExtension [plugin-id extension {:keys [global-settings encryptor] :as context}]
  (try
    (let [name (.getName extension)
          description (.getDescription extension)
          settings (settings extension encryptor global-settings)]
      (reify ConfigurationExtension
        (-id [this] plugin-id)
        (-name [this] name)
        (-description [this] description)
        (-stored-settings [this] (stored-or-default-settings plugin-id settings context))
        (-store-settings [this values] (store-settings plugin-id name extension context values))))
    (catch Exception e
      (let [msg (format "Failed to create extension in plugin %s" plugin-id)]
        (log/warnf e msg)))))

(comment
  (def db (let [db (atom {})]
            (reify db/SettingsDb
              (-update-settings [_ _ name settings] (swap! db assoc :settings settings))
              (-settings-for [_ _] (get @db :settings [])))))
  (def context {:db db
                :encryptor (codescene.crypto.encryptor/->NoEncryption)
                :global-settings []})
  (def ext (reify com.codescene.plugin.ProjectConfigurationExtensionPoint
               (getName [_] "dummy-config")
               (getDescription [_] "dummy-desc")
               (getSettings [_ _context] [(proxy [com.codescene.plugin.configuration.TextSetting] []
                                            (getName [] "Dummy Setting")
                                            (getId [] "dummy-setting")
                                            (getCaption [] "Dummy Caption")
                                            (getValue [] "Dummy Value"))])
             (validateSettings [_ _context _values] [])))
  (def e (->ConfigurationExtension "dummy-plugin" ext context))
  (_ce/store-settings e [{:name "updated-name", :id "dummy-setting" :value "updated-value"}])
  (_ce/stored-settings e))

