(ns codescene.features.factors.kpi-time-series
  (:require
    [clj-time.core :as t]
    [clj-time.format :as f]
    [clojure.java.io :as io]
    [clojure.java.jdbc :as jdbc]
    [codescene.analysis.paths :as paths]
    [codescene.factors.core :refer [kpi-type->path kpi-types]]
    [codescene.features.dashboard.core :as dashboard-common]
    [codescene.features.util.maps :as maps]
    [codescene.util.csv-conversions :as csv-conversions]
    [codescene.util.json :as json]
    [hugsql.core :as hugsql]
    [taoensso.encore :as encore])
  (:import
    java.io.FileNotFoundException))

(hugsql/def-db-fns "codescene/features/factors/kpi-time-series.sql")
(declare time-series-by-project-id*)
(declare insert-time-series*)
(declare delete-time-series*)
(declare time-series-by-analysis-id*)

(defn analysis->time-series-record [project-id analysis-id results-absolute-path kpi-type]
  (try
    {:kpi-time-series (seq (json/read (str results-absolute-path "/" (kpi-type kpi-type->path))))
     :kpi-type (name kpi-type)
     :project-id project-id
     :analysis-id analysis-id}
    (catch FileNotFoundException _ nil)))

;; (s/def ::kpi-time-series-data (s/coll-of (s/keys :req-un [::date ::type])))
;; (s/def ::kpi-time-series-record (s/keys :req-un [::project-id ::analysis-id ::kpi-type ::kpi-time-series]))

(defn ->insertable [kpi-time-series-record]
  (-> kpi-time-series-record
      (update :kpi-time-series json/generate-string)))

(defn declob
  "Turn a clob into a String"
  [clob]
  (when clob                                                ;; nil / NULL in db
    (if (string? clob)
      clob
      (slurp (.getCharacterStream clob)))))

(defn ->readable [kpi-time-series-from-db]
  (-> kpi-time-series-from-db
      maps/->kebab
      (update :kpi-time-series declob)
      (update :kpi-time-series json/parse-string)
      (update :kpi-type keyword)))

(defn replace-time-series
  [db-spec time-series-record]
  (jdbc/with-db-transaction [tx db-spec]
                            (delete-time-series* tx time-series-record)
                            (insert-time-series* tx time-series-record)))

;; (s/fdef save-time-series
;;   :args (s/cat :this ::kpi-time-series-db
;;                :absolute-analysis-path ::cs-specs/non-empty-string))
(defn update-time-series [db-spec project-id analysis-id absolute-analysis-path]
  (doall (->> kpi-types
              (map (partial analysis->time-series-record project-id analysis-id absolute-analysis-path))
              (map ->insertable)
              (map (partial replace-time-series db-spec)))))

(defn time-series-by-project-id [db-spec project-id]
  (map ->readable (time-series-by-project-id* db-spec {:project-id project-id})))

(defn time-series-by-analysis-id
  [db-spec project-id analysis-id]
  (into {}
        (comp
         (map ->readable)
         (map (juxt :kpi-type :kpi-time-series)))
        (time-series-by-analysis-id* db-spec {:project_id project-id :analysis_id analysis-id})))

(defn- by-kpi-type [all-kpis kpi-type]
  (hash-map kpi-type
            (some (fn [kpi] (when (= kpi-type (:kpi-type kpi)) kpi)) all-kpis)))

(defn all-kpis-for-project!
  ;; TODO @aasa spec data format
  "Returns a single map of kpi-type the corresponding time series data:
  {:code-health [{:date ... :kpi ... } ... n]
   :delivery ... n}."
  [db-spec project-id]
  (let [all-kpis (time-series-by-project-id db-spec project-id)
        kpi-types (map :kpi-type all-kpis)]
    (into {}
          (map
           (partial by-kpi-type all-kpis)
           kpi-types))))

(defn- remove-delivery-details
  [delivery-kpi]
  (-> delivery-kpi
      (update-in [:kpis :day] select-keys  [:kpi :sub-kpis])
      (update-in [:kpis :week] select-keys [:kpi :sub-kpis])
      (update-in [:kpis :month] select-keys [:kpi :sub-kpis])
      (update-in [:kpis :year] select-keys  [:kpi :sub-kpis])))

(defn- remove-sub-kpi-name
  [kpi-time-series]
  (update kpi-time-series :sub-kpis (fn [sub-kpis]
                                      (map #(dissoc % :name) sub-kpis))))

(defn- remove-details
  [kpi-time-series]
  (map
   (fn pruning [{:keys [kpis] :as i}]
     (cond-> (select-keys i [:date :kpi :sub-kpis :kpis :error :delta])
       kpis remove-delivery-details
       true remove-sub-kpi-name))
   kpi-time-series))

(defn remove-error-data-points
  "Remove all data points that are not valid.

  To display misconfiguration we instead use the config diagnostic object, rather than
  by reading error data points."
  [kpi-time-series]
  (remove #(contains? % :error) kpi-time-series))

(def iso-date-formatter (f/formatter "yyyy-MM-dd"))

(def date->string (partial f/unparse iso-date-formatter))

(defn- time-series-start-date
  "The idea is to choose a start date that gives us about a month worth of data.
  But we also include an extra datapoint before this cut-off date, because
  we don't want the graph to appear cut off."
  [time-series-dates]
  (encore/when-let [last-date (some-> time-series-dates last f/parse)
                    approx-start-date (date->string (t/minus last-date (t/months 1)))
                    [before after] (split-with #(neg? (compare % approx-start-date)) time-series-dates)]
     (cond
       (> (count before) 0) (last before)
       :else (first after))))

(defn- ->date-series
  [kpi-time-series]
  (sort (map :date kpi-time-series)))

(defn select-data-points-in-scope
  "Selects a slice of the time series by deriving a start and end date
  from the last element.

  Other values are removed, taking into account not everything in the time
  series may be in order due to historical mishaps.

  Comparisons of dates are done alphabetically."
  [kpi-time-series]
  (encore/when-let [sorted-dates (->date-series kpi-time-series)
                    start-date (time-series-start-date sorted-dates)
                    end-date (last sorted-dates)]
    (->> kpi-time-series
         (remove #(pos? (compare (:date %) end-date)))
         (remove #(neg? (compare (:date %) start-date))))))

(defn prune-project-keys-for-projects-page
  [project]
  ;; TODO: this might not be the best strategy for narrowing
  ;; the data here. For now, this helps avoid some
  ;; unnecessary CLOBs in on-prem that sneak into the project config.
  (select-keys project [:name
                        :description
                        :id
                        :last-analysis
                        :analysis-data
                        :analysis-destination
                        :groups
                        :group-id
                        :run-status
                        :pr-integration]))

(defn- total-lines-of-code
  [analysis-directory]
  (let [csv-file (io/file analysis-directory paths/dashboard-csv)]
    (dashboard-common/application-code-loc csv-file)))

(defn- active-authors [analysis-directory]
  (let [csv-file (io/file analysis-directory paths/dashboard-csv)
        dashboard-data (csv-conversions/optional-csv->map csv-file)]
    (:activeauthors dashboard-data)))

(defn kpis-for-projects-page
  "Analysis job has keys :id, :analysis-time, :dir."
  [project analysis-job-with-dir kpis]
  (-> project
      (assoc :last-analysis (assoc (select-keys analysis-job-with-dir [:id :analysis-time])
                              :total-lines-of-code (total-lines-of-code (:dir analysis-job-with-dir))
                              :active-authors (active-authors (:dir analysis-job-with-dir))))
      (assoc :analysis-data kpis)))
