(ns codescene.features.config.properties
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [codescene.presentation.display :refer [->maybe-int]]
            [codescene.features.api.spec.analysis :as api-specs]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.specs :as specs]
            [medley.core :as m]))

(s/def ::ui-param keyword?)
(s/def ::param-read ifn?)
(s/def ::deser ifn?)
(s/def ::encrypt? boolean?)
(s/def ::missing-false? boolean?)
(s/def ::fix-fn (s/nilable ifn?))
(s/def ::default any?)
(s/def ::type-info (s/keys :opt-un [::missing-false? ::encrypt? ::param-read ::deser]))
(s/def ::property (s/keys :req-un [::ui-param ::default ::type-info]
                          :opt-un [::fix-fn]))
(s/def ::properties (s/map-of any? ::property))

(def standard-param-fix str/trim)

;; param-read fn is (fn [property-map data] v)
(defn default-param-read [{:keys [ui-param fix-fn] :or {fix-fn standard-param-fix}} data]
  (if-some [e (find data ui-param)] (fix-fn (val e)) ::missing))

(defn file-read [{:keys [ui-param]} params]
  (or (m/find-first seq [(some-> params (get-in [(keyword (str (name ui-param) "-file")) :tempfile]) slurp)
                         (get params ui-param)])
      ::missing))

(defn ival [] {:deser #(when % (->maybe-int %)) :api-spec integer?})

(defn file-string
  [encrypt?] {:encrypt? encrypt? :api-spec ::specs/non-empty-string :param-read file-read})

(defn enum
  [s] {:api-spec s })

(defn sval
  "Non-empty string value."
  ([] (sval false))
  ([encrypt?] {:encrypt? encrypt? :api-spec ::specs/non-empty-string}))

(defn sval-
  "String value that can be empty."
  ([] (sval false))
  ([encrypt?] {:encrypt? encrypt? :api-spec string?}))

(defn bool [] {:deser #(case %
                         nil nil
                         ("true" true "1") true
                         ("false" false "0") false)
               :missing-false? true
               :api-spec ::api-specs/boolean-string})

(defn property
  ([id ui-param type-info]
   (property id ui-param type-info nil))
  ([id ui-param type-info default]
   (map-of id ui-param default type-info)))

(s/fdef property
        :args (s/cat :id any? :ui-param ::ui-param :type-info ::type-info :default (s/? ::default)))

(defn from-req-params
  "Reads HTTP form params into an option -> value map.

  When data comes from UI, then html-form-input? should be set to true, since in UI
  boolean values are implemented with checkboxes which only send parameter when checked."
  [properties params try-encrypt html-form-input?]
  (m/map-vals
    (fn [{:keys [default type-info] :as om}]
      (let [{:keys [param-read deser encrypt? missing-false?] :or {param-read default-param-read deser identity}} type-info
            v (param-read om params)]
        (if (= v ::missing)
          (if (and html-form-input? missing-false?) false default)
          (cond-> (deser v) encrypt? try-encrypt))))
    (m/index-by :id properties)))

(defn to-params
  [properties data]
  (let [by-id (m/index-by :id properties)]
    (m/map-keys #(get-in by-id [% :ui-param] %) data)))

(defn from-db
  "Reads DB config into a map."
  [properties data]
  (m/map-kv-vals
    (fn [option {:keys [default type-info]}]
      (if-some [v (find data option)] ((or (:deser type-info) identity) (val v)) default))
    (m/index-by :id properties)))