(ns codescene.features.components.auth
  (:require [clojure.spec.alpha :as s]
            [codescene.features.spec.auth :as auth]
            [codescene.features.spec.commons :as commons-spec]))

(defprotocol AuthComponent
  (-user [this request])
  (-process-required-privileges [this required-privileges])
  (-has-project-access? [this request project-id])
  (-has-devtools-access? [this request project-id])
  (-has-refactoring-access? [this request])
  (-can-use-feature? [this request feature]))

(s/fdef user
        :args (s/cat :this some?
                     :request map?)
        :ret (s/nilable ::auth/user))
(defn user
  "return authenticated user from request,
  onprem is using friend and the user is extracted from session identity,
  cloud is using buddy and the user is taken from request identity"
  [this request]
  (-user this request))

(s/fdef process-required-privileges
        :args (s/cat :this some?
                     :required-privileges (s/coll-of string?))
        :ret (s/and set? (s/coll-of string?)))
(defn process-required-privileges
  "process required privileges by adding connected privileges depending on product,
  for example in on-prem we add the privileges for each group"
  [this required-privileges]
  (-process-required-privileges this required-privileges))

(s/fdef has-project-access?-privileges
        :args (s/cat :this some?
                     :request map?
                     :project-id ::commons-spec/id)
        :ret boolean?)
(defn has-project-access?
  "return true if autenticated user have access to project id"
  [this request project-id]
  (-has-project-access? this request project-id))

(s/fdef has-devtools-access?
  :args (s/cat :this some?
               :request map?
               :project-id ::commons-spec/id)
  :ret boolean?)

(defn has-devtools-access?
  "return true if autenticated user can access project via devtools (vscode/cli)"
  [this request project-id]
  (-has-devtools-access? this request project-id))

(defn has-refactoring-access?
  "return true if autenticated user can use the refactoring service"
  [this request]
  (-has-refactoring-access? this request ))

(s/fdef can-use-feature?
  :args (s/cat :this some?
               :request map?
               :feature keyword?)
  :ret (s/nilable boolean?))

(defn can-use-feature?
  "return true if authenticated user's license has access to the specified feature,
  onprem defines restricted-features
  cloud defines plan-features"
  [this request feature]
  (-can-use-feature? this request feature))
