(ns codescene.features.code-coverage.parsers.open-cover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :sequence-point-coverage
                                  :method-coverage 
                                  :branch-coverage])

(defn- covered-seq-point? [{:keys [attrs] :as line-el}]
  (some-> attrs :vc Integer/parseInt pos?))

(defn- line-num [{:keys [attrs] :as line-el}]
  (:sl attrs))

(defn- branch-coverage
  [{:keys [numBranchPoints visitedBranchPoints] :as _summary}]
  (when (some-> numBranchPoints pos?)
    {:covered visitedBranchPoints
     :total numBranchPoints
     :coverage (/ visitedBranchPoints numBranchPoints)}))

(defn- method-coverage
  [{:keys [numMethods visitedMethods] :as _summary}]
  (when (some-> numMethods pos?)
    {:covered visitedMethods
     :total numMethods
     :coverage (/ visitedMethods numMethods)}))

(defn- sequence-point-coverage
  [{:keys [numSequencePoints visitedSequencePoints] :as _summary}]
  (when (some-> numSequencePoints pos?)
    {:covered visitedSequencePoints
     :total numSequencePoints
     :coverage (/ visitedSequencePoints numSequencePoints)}))

(defn- line-coverage
  [seq-point-els]
  (let [lines (->> seq-point-els (map line-num) distinct count)
        covered (->> seq-point-els (filter covered-seq-point?) (map line-num) distinct count)]
    (when (pos? lines)
      {:total lines
       :covered covered
       :coverage (/ covered lines)})))

(defn- parse-file-el [->summary ->seq-points {:keys [attrs] :as _el}]
  (let [uid (:uid attrs)
        path (some-> (:fullPath attrs)
                     (str/replace "\\" "/"))
        summary (->summary uid)
        seq-points (->seq-points uid)]
    (m/assoc-some {:path              path
                   :name              (-> path (str/split #"/") last)}
                 :branch-coverage (branch-coverage summary)
                 :method-coverage (method-coverage summary)
                 :sequence-point-coverage (sequence-point-coverage summary)
                 :line-coverage (line-coverage seq-points))))

 (def used-summary-key?
   #{:numSequencePoints :visitedSequencePoints
     :numBranchPoints :visitedBranchPoints
     :numMethods :visitedMethods})

(defn- merge-summary-attrs
  [acc method-el]
  (let [file-uid (-> (utils/sub-node :FileRef method-el) :attrs  :uid)
        attrs (->> (utils/sub-node :Summary method-el)
                   :attrs
                   (filter (fn [[k _v]] (used-summary-key? k))) ;; Keep only used keys
                   (#(for [[k v] %] [k (Integer/parseInt v)]))  ;; parse values as ints
                   (into {}))]
    (update acc file-uid #(merge-with + % attrs))))

(defn- ->summary-lookup [method-els]
  (reduce merge-summary-attrs {} method-els))

(defn- ->seq-points-lookup [seq-point-els]
  (group-by #(-> % :attrs :fileid) seq-point-els))
 
(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        files (utils/sub-nodes-in xml [:Modules :Module :Files :File])
        methods (utils/sub-nodes-rec :Method xml)
        seq-points (utils/sub-nodes-rec :SequencePoint xml)
        ->seq-points (->seq-points-lookup seq-points)
        ->summary (->summary-lookup methods)]
    (map (partial parse-file-el ->summary ->seq-points) files)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "open-cover")
    (-name [this] "OpenCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/OpenCover_altcover.xml")
  (read-coverage f)
  )
