(ns codescene.features.code-coverage.parsers.open-clover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :method-coverage
                                  :condition-coverage
                                  :statement-coverage])

(defn- covered-line? [{:keys [attrs] :as line-el}]
  (let [{:keys [count falsecount truecount]} attrs
        pos-count? #(some-> % Integer/parseInt pos?)]
    (some pos-count? [count falsecount truecount])))

(defn- line-num [{:keys [attrs] :as line-el}]
  (:num attrs))

(defn attrs->coverage [{:keys [attrs] :as _el} covered-attr total-attr]
  (let [covered-str (get attrs covered-attr)
        total-str (get attrs total-attr)]
    (when (and covered-str total-str)
      (let [covered (Integer/parseInt covered-str)
            total (Integer/parseInt total-str)]
        (when (pos? total)
          {:covered covered
           :total total
           :coverage (/ covered total)})))))

(defn- method-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredmethods :methods))

(defn- condition-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredconditionals :conditionals))

(defn- statement-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredstatements :statements))

(defn- line-coverage [line-els]
  (let [lines (->> line-els (map line-num) distinct count)
        covered (->> line-els (filter covered-line?) (map line-num) distinct count)]
    (when (pos? lines)
      {:covered covered
       :total lines
       :coverage (/ covered lines)})))

(defn parse-file-el [{:keys [attrs] :as el}]
  (let [path (-> (:path attrs)
                 (str/replace "\\" "/"))
        name (:name attrs)
        metrics (utils/sub-node :metrics el)
        lines (utils/sub-nodes :line el)]
    ;; See also http://openclover.org/doc/manual/4.2.0/faq--how-are-coverage-percentages-calculated.html
    (m/assoc-some {:path path
                   :name name}
                  :method-coverage (method-coverage metrics)
                  :condition-coverage (condition-coverage metrics)
                  :statement-coverage (statement-coverage metrics)
                  :line-coverage (line-coverage lines))))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        file-els (utils/sub-nodes-in xml [:project :package :file])
        test-file-els (utils/sub-nodes-in xml [:test-project :package :file])]
    (map parse-file-el (concat file-els test-file-els))))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "open-clover")
    (-name [this] "OpenClover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/clover.xml")
  (read-coverage f)
  )