(ns codescene.features.code-coverage.parsers.ncover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage :sequence-point-coverage])

(defn- parse-path-and-counts 
  [[path {:keys [lines-total lines-covered seqp-total seqp-covered]}]]
  (let [forward-slashed-path (str/replace path "\\" "/")]
    (m/assoc-some {:path                    forward-slashed-path
                   :name                    (-> forward-slashed-path (str/split #"/") last)}
                  :line-coverage (when (pos? lines-total)
                                   {:covered lines-covered
                                    :total lines-total
                                    :coverage (/ lines-covered lines-total)})
                  :sequence-point-coverage (when (pos? seqp-total)
                                             {:covered seqp-covered
                                              :total seqp-total
                                              :coverage (/ seqp-covered seqp-total)}))))

(defn- parse-entries
  "Parses entries and returns two sets [covered all]"
  [parse-value-fn coll]
  (reduce (fn [[covered all] value]
            (let [{:keys [id covered?]} (parse-value-fn value)]
              [(cond-> covered covered? (conj id))
               (conj all id)]))
          [#{} #{}]
          coll))

(defn- seqpnt-covered?
  [{{:keys [visitcount]} :attrs :as _seqpnt-el}]
  (-> visitcount Integer/parseInt pos?))

(defn- seqpnt->line [{:keys [attrs] :as _seqpnt-el}]
  (:line attrs))

(defn- seqpnt->document [{:keys [attrs] :as _seqpnt-el}]
  (:document attrs))

(defn- counts
  [seqpnt-els]
  (let [covered-seq-pnts (filter seqpnt-covered? seqpnt-els)]
    {:lines-total (->> seqpnt-els (map seqpnt->line) distinct count)
     :lines-covered (->> seqpnt-els (filter seqpnt-covered?) (map seqpnt->line) distinct count)
     :seqp-total (count seqpnt-els)
     :seqp-covered (count covered-seq-pnts)}))

(defn- ->counts-lookup [seqpnts]
  (->> seqpnts
       (group-by seqpnt->document)
       (m/map-vals counts)))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :nntamespace-aware false :support-dtd false))
        seqpnts (utils/sub-nodes-rec :seqpnt xml)
        path->counts (->counts-lookup seqpnts)]
    (map parse-path-and-counts path->counts)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "ncover")
    (-name [this] "NCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/NCover1.5.8.xml")
  (read-coverage f)
  )