(ns codescene.features.code-coverage.parsers.lcov-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :branch-coverage
                                  :function-coverage])

(defn- end-of-record? [l]
  (str/starts-with? l "end_of_record"))

(defn- ->record [lines]
  (->> lines
       (mapv #(str/split % #":" 2))
       (group-by first)
       (m/map-vals (partial map second))))

(defn- parse-entries
  "Parses entries and returns two sets [covered all]"
  [parse-value-fn coll]
  (reduce (fn [[covered all] value]
            (let [{:keys [id covered?]} (parse-value-fn value)]
              [(cond-> covered covered? (conj id))
               (conj all id)]))
          [#{} #{}]
          coll))

(defn- parse-lines
  "Parses DA entries abd returns two sets [covered all]"
  [da-coll]
  (parse-entries (fn [value]
                   (let [[line hits] (str/split value #"," 2)]
                     {:id line
                      :covered? (some-> hits Integer/parseInt pos?)}))
                 da-coll))

(defn- parse-branches
  "Parses BRDA entries and returns two sets [covered all]"
  [bdra-coll]
  (parse-entries (fn [value]
                   (let [[line block branch taken] (str/split value #"," 4)]
                     {:id [line block branch]
                      :covered? (nil? (#{"-" "0"} taken))}))
                 bdra-coll))

(defn- parse-single-integer-value
  "Parses single integer value, the first element will be used if value coresponding to the key is a collection"
  [m key]
  (let [v (get m key)
        e (cond-> v
                  (coll? v) first)]
    (cond
      (or (nil? e) (number? e)) e
      (re-matches #"\d*" e) (Integer/parseInt e))))

(defn ->entry [covered total]
  (when (and covered total (> total 0))
    {:covered  covered
     :total    total
     :coverage (/ covered total)}))

(defn- record->coverage-entry [r]
  ;; See also https://github.com/linux-test-project/lcov/issues/113
  (let [SF (-> (get r "SF") first (str/replace "\\" "/"))
        line-data (parse-lines (get r "DA"))
        branch-data (parse-branches (get r "BRDA"))
        ;; the LCOV file contains for each file entry the covered number of functions "FNH" and the total number of functions "FNF"
        ;; https://github.com/linux-test-project/lcov/blob/master/man/geninfo.1#L1501
        function-data (->entry (parse-single-integer-value r "FNH") (parse-single-integer-value r "FNF"))
        ->coverage (fn [[covered all]]
                     (when (seq all)
                       (->entry (count covered) (count all))))]
    (m/assoc-some {:path SF
                   :name (-> SF (str/split #"/") last)}
                  :line-coverage (->coverage line-data)
                  :branch-coverage (->coverage branch-data)
                  :function-coverage function-data)))

(defn- records
  [lines]
  (when (seq lines)
    (let [record (->> (take-while (complement end-of-record?) lines) ->record)
          rest (->> (drop-while (complement end-of-record?) lines) (drop 1))]
      (lazy-seq
       (cons record
             (records rest))))))

(defn- read-coverage* [r]
  (->> (line-seq r)
       records
       (map record->coverage-entry)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "lcov")
    (-name [this] "LCov")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/lcov.info")
  (->> (read-coverage f))
  )
