(ns codescene.features.code-coverage.parsers.jacoco-parser
  "DTD: https://github.com/jacoco/jacoco/blob/master/org.jacoco.report/src/org/jacoco/report/xml/report.dtd"
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [medley.core :as m]
            [clojure.string :as str]))

(def ^:private supported-metrics [:line-coverage
                                  :branch-coverage])

(defn- extract-counter-attribute
  [el type attribute]
  (some
   (fn [{:keys [attrs]}]
     (when (= type (:type attrs))
       (get attrs attribute)))
   (utils/sub-nodes :counter el)))

(defn safe-parse-int
  "Parse a string into an int, allowing for nil"
  [str]
  (if (nil? str)
    nil
    (Integer/parseInt str)))

(defn- calculate-coverage [covered-str missed-str]
  (let [covered (safe-parse-int covered-str)
        missed (safe-parse-int missed-str)]
    (when (not-any? nil? [covered missed])
      (/ covered (+ covered missed)))))

(defn- coverage-data [el type]
  (let [covered (-> el (extract-counter-attribute type :covered) safe-parse-int)
        missed (-> el (extract-counter-attribute type :missed) safe-parse-int)]
     (when (not-any? nil? [covered missed])
       {:covered covered
        :total (+ covered missed)
        :coverage (/ covered (+ covered missed))})))

(defn- line-coverage [el]
  (coverage-data el "LINE"))

(defn- branch-coverage [el]
  (coverage-data el "BRANCH"))

(defn- parent-path
  "Returns the path consisting of names of the parent elements (presumably 'group' elems)
  joined by '/'.
  Returns nil if no parent's are given
  or a path ending with `/` signaling a folder."
  [parents]
  (->> parents
       (filter #(= :group (:tag %)))
       ;; append the slash here to produce trailing slash
       (map #(-> % :attrs :name (str "/")))
       ;; the %s is added to be able to inject the supath later in between parent and code coevareg file path
       (#(when (seq %) (concat % ["%s"])))
       str/join))

(defn- parse-source-el [package-name parents {:keys [attrs] :as el}]
  ;; TODO: extract coverage from "COMPLEXITY", "METHOD" and "CLASS" too ?
  (let [file-name (:name attrs)]
    (m/assoc-some {:path (format "%s%s/%s" (parent-path parents) package-name file-name)
                   :name file-name}
                  :branch-coverage (branch-coverage el)
                  :line-coverage (line-coverage el))))

(defn- parse-package-el [{:keys [attrs parents] :as el}]
  (let [package-name (:name attrs)
        source-file-els (utils/sub-nodes :sourcefile el)]
    (map (fn [source-file] (parse-source-el package-name parents source-file))
         source-file-els)))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        ;; recursively search for 'package' elements because they can be nested inside 'group'
        package-els (utils/sub-nodes-rec :package xml)]
    (mapcat parse-package-el package-els)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "jacoco")
    (-name [this] "JaCoCo")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/Java/JaCoCo0.8.3.xml")
  (def f "/Users/jumar/workspace/CODESCENE/CODE/codescene/features/test/codescene/features/code_coverage/testdata/Java/JaCoCo0.8.3_groups.xml")
  (->> (read-coverage f))

  ;;
  )

