(ns codescene.features.code-coverage.parsers.dotcover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [clojure.set :as set]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage :statement-coverage])

(defn- covered-statement? [{:keys [attrs] :as stmt-el}]
  (= "True" (:Covered attrs)))

(defn- statement->line [{:keys [attrs] :as stmt-el}]
  (:Line attrs))

(defn- coverage [n-covered n-total]
  (when (pos? n-total)
    {:covered n-covered
     :total n-total
     :coverage (/ n-covered n-total)}))

(defn statement-coverage [statement-els]
  (let [covered (filter covered-statement? statement-els)]
    (coverage (count covered) (count statement-els))))

(defn line-coverage [statement-els]
  (let [lines (->> statement-els (map statement->line) distinct)
        covered (->> statement-els (filter covered-statement?) (map statement->line) distinct)]
    (coverage (count covered) (count lines))))

(defn- parse-file-el [file->statements {:keys [attrs] :as file-el}]
  (let [{name :Name index :Index} attrs
        path (str/replace name "\\" "/")
        statements (file->statements index)]
    (m/assoc-some {:path path
                   :name (-> path (str/split #"/") last)}
                  :statement-coverage (statement-coverage statements)
                  :line-coverage (line-coverage statements))))

(defn- by-file-index
  [statements]
  (->> statements
       (group-by (comp :FileIndex :attrs))
       (into {})))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        files (utils/sub-nodes-in xml [:FileIndices :File])
        file->statements (-> (utils/sub-nodes-rec :Statement xml)
                             by-file-index)]
    (map (partial parse-file-el file->statements) files)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "dotcover")
    (-name [this] "DotCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/dotCover.xml")
  (read-coverage f)
  )
