(ns codescene.features.code-coverage.parsers.cobertura-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.set :as set]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage])

(defn- parse-path-and-lines [[path {:keys [covered total]}]]
  (let [forward-slashed-path (str/replace path "\\" "/")
        n-total (count total)]
    (m/assoc-some {:path          forward-slashed-path
                   :name          (-> forward-slashed-path (str/split #"/") last)}
                  :line-coverage (when (pos? n-total)
                                   {:covered (count covered)
                                    :total n-total
                                    :coverage (when (pos? n-total)
                                                (/ (count covered) n-total))}))))

(defn- line-numbers [line-els]
  (->> line-els
       (map :attrs)
       (map :number)))

(defn- merge-lines 
  "Merge sets of covered/uncovered line nbrs into accumulated map indexed by filename"
  [acc {:keys [attrs] :as class-el}]
  (let [filename (:filename attrs)
        lines (utils/sub-nodes-in class-el [:lines :line])
        covered (->> lines
                     (filter #(not= "0" (-> % :attrs :hits)))
                     line-numbers)
        total (line-numbers lines)]
    (update acc filename #(merge-with set/union % {:covered (set covered)
                                                   :total (set total)}))))

(defn- ->lines-lookup [class-locs]
  (reduce merge-lines {} class-locs))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        classes (utils/sub-nodes-in xml [:packages :package :classes :class])
        path->lines (->lines-lookup classes)]
    (map parse-path-and-lines path->lines)))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "cobertura")
    (-name [this] "Cobertura")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/Java/Cobertura2.1.1.xml")
  (read-coverage f)
  )