(ns codescene.features.code-coverage.parsers.bullseye-parser
  (:require [codescene.features.code-coverage.parsers.xml-utils :refer [filter-on-tag]]
            [codescene.features.code-coverage.parser :refer [Parser]]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :method-coverage
                                  :condition-decision-coverage
                                  :decision-coverage])

(defn attrs->coverage [attrs covered-attr total-attr]
  (let [covered-str (get attrs covered-attr)
        total-str (get attrs total-attr)]
    (when (and covered-str total-str)
      (let [covered (Integer/parseInt covered-str)
            total (Integer/parseInt total-str)]
        (when (pos? total)
          (/ covered total))))))

(defn attrs->coverage-data [attrs covered-attr total-attr]
  (let [covered-str (get attrs covered-attr)
        total-str (get attrs total-attr)]
    (when (and covered-str total-str)
      (let [covered (Integer/parseInt covered-str)
            total (Integer/parseInt total-str)]
        (when (pos? total)
          {:covered covered
           :total total
           :coverage (/ covered total)})))))

(defn- condition-decision-coverage [attrs]
  (attrs->coverage-data attrs :cd_cov :cd_total))

(defn- decision-coverage [attrs]
  (attrs->coverage-data attrs :d_cov :d_total))

(defn- method-coverage [attrs]
  (attrs->coverage-data attrs :fn_cov :fn_total))

(defn- line-coverage
  "we count as total every line probe and if the event is not 'none' then the line is counted as covered
   see: https://github.com/SonarOpenCommunity/sonar-cxx/wiki/sonar.cxx.bullseye.reportPaths
   and https://github.com/SonarOpenCommunity/sonar-cxx/blob/95ea2aec5d0df9374435ae89bd12ffde8734e7a2/cxx-sensors/src/main/java/org/sonar/cxx/sensors/coverage/bullseye/BullseyeParser.java"
  [src-el]
  (let [fn-probe-attrs (->> (:content src-el)
                            (filter-on-tag :fn)
                            (mapcat :content)
                            (filter-on-tag :probe)
                            (map :attrs))
        distinct-count-as-string (comp str count distinct)
        ln-covered (->> (filter #(not= "none" (:event %)) fn-probe-attrs) (map :line) distinct-count-as-string)
        ln-total (->> (map :line fn-probe-attrs) distinct-count-as-string)]
    (attrs->coverage-data {:ln_covered ln-covered :ln_total  ln-total} :ln_covered :ln_total)))

(defn ->coverage-entry [folders {:keys [attrs] :as src-el}]
  (let [{:keys [name]} attrs]
  (m/assoc-some {:path (str/join "/" (conj folders name))
                   :name name}
                  :condition-decision-coverage (condition-decision-coverage attrs)
                  :decision-coverage (decision-coverage attrs)
                  :method-coverage (method-coverage attrs)
                  :line-coverage (line-coverage src-el))))

(defn- parse-folder-el [parent-folders {:keys [attrs content] :as _folder-el}]
  (let [{:keys [name]} attrs
        folders (conj parent-folders name)
        src-els (filter-on-tag :src content)
        sub-folder-els (filter-on-tag :folder content)]
    (concat (map (partial ->coverage-entry folders) src-els)
            (mapcat (partial parse-folder-el folders) sub-folder-els))))

(defn- read-coverage* [reader]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))]
    (->> xml :content
         (filter-on-tag :folder)
         (mapcat #(parse-folder-el [] %)))))

(defn- read-coverage [f]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader] (read-coverage* reader))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "bullseye")
    (-name [this] "BullsEye")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/bullseye.xml")
  (read-coverage f)
  )
