(ns codescene.features.client.oauth2
  (:require [buddy.core.codecs :as codecs]
            [buddy.core.nonce :as nonce]
            [codescene.features.util.url :as url-utils]
            [codescene.util.data :as data]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(defprotocol OAuth2Provider
  "Protocol for participation in the standard Authorization Code based OAuth2 flow."
  (oauth2-provider-id [this] "Returns provider ID keyword for this provider.")
  (oauth2-scope [this params] "Returns scope string (if any)")
  (oauth2-param-gen [this] "Returns OAuth2GenGrantParameters")
  (oauth2-provider-timeout [this] "Returns timeout in ms for requests to this provider, can be nil.")
  (oauth2-mount-point [this]
    "Vector like [http://localhost:4000 /login/github],
    So the host connection string + relative URI of where the /authorize /callback points are found in CodeScene.")
  (oauth2-spec [this]
    "Returns information for OAuth2 process as a map with the following keys:
    - authorize-url: Base (!) URL of the authorization endpoint
    - access-token-url: Base (!) URL of the access token endpoint"))

(defn rand32
  "40-byte nonce as URL safe base64 string. Nonce used 8 bytes of time (ms) data, and 32-byte of random."
  [] (codecs/bytes->b64-str (nonce/random-bytes 32) true))

(defrecord AuthFlow [session-id state code-verifier next-url])

(defn init-auth-flow
  "Creates new state and code verifier, stores them in kv-store with redirect url. Returns AuthFlow.

  The state acts as am identifier of OAuth2 flow invocation. It is supplied by AS on redirect, so we can link first request
  and subsequent request."
  [{:keys [headers params] :as r}]
  (let [state (rand32)
        verifier (rand32)
        session-id (or (:session/key r) (throw (Exception. "Cannot find Ring Session Cookie")))
        next-uri (or (:next params)
                     ;; if next URI is not specified, redirect to the page that initiated
                     (headers "referer")
                     "/")]
    (->AuthFlow session-id state verifier (url-utils/try-relative-url next-uri))))

(defn throw-error
  [message provider-name]
  (throw (ex-info (format "Invalid OAuth2 callback for provider %s, %s." provider-name message) {})))

(defn assert-callback-params!
  "Validates a few common OAuth2 error scenarios. If everything is OK, nil is returned,
  otherwise it returns whatever authorization failure returns."
  [{:keys [error error_description code state] :as _params} provider-name]
  (when (or error_description error)
    (throw-error (format "(%s, %s)" error error_description) provider-name))
  (when (empty? code)
    (throw-error "no code received" provider-name))
  (when (empty? state)
    (throw-error "no 'state' received" provider-name)))

(defn assert-valid-flow!
  "Throws an exception if state value is not an active nonce with the right session.

  Returns auth-flow."
  [{:keys [session-id] :as flow} curr-session-id provider-name]
  (when (nil? flow)
    (throw-error "no OAuth flow started" provider-name))
  (when (not= curr-session-id session-id)
    (log/warnf "OAuth2 session mismatch, init session='%s', callback session='%s'" session-id curr-session-id)
    (throw-error "session that started OAuth flow and current session are not the same" provider-name))
  flow)

(defprotocol OAuth2GenGrantParameters
  "Protocol for generating form params when doing the authorization flow.

  These need not include state, redirect_uri and such."
  (authorize-params [this]
    "Returns response_type, client_id,... to start the authorization flow.")
  (base-grant-params [this]
    "Basic params used in all grant calls.")
  (code-grant-params [this code]
    "The standard authorize flow grant parameters.")
  (refresh-grant-params [this refresh-token]
    "The refresh token authorize flow grant parameters.")
  (client-credentials-params [this]
    "The client_credentials authorize flow grant parameters.
    This typically requires a confidential/private OAuth2 app."))

(defrecord BasicGenGrantParameters [client-id client-secret scope]
  OAuth2GenGrantParameters
  (authorize-params [this]
    {:client_id client-id
     :response_type "code"})
  (base-grant-params [this]
    {:client_id client-id
     :client_secret client-secret})
  (code-grant-params [this code]
    (m/assoc-some {:code code :grant_type "authorization_code"} :scope scope))
  (refresh-grant-params [this refresh-token]
    {:refresh_token refresh-token
     :grant_type "refresh_token"})
  (client-credentials-params [this]
    (m/assoc-some {:grant_type "client_credentials"} :scope scope)))

(defn oauth2-mount-point->callback-url
  [oauth2-mount-point]
  (let [[base-url relative-uri] oauth2-mount-point]
    (str base-url relative-uri "/callback")))

(defn callback-url
  "Generate callback URL for Provider to call for the second part of Authentication flow."
  [oauth2-provider]
  (oauth2-mount-point->callback-url (oauth2-mount-point oauth2-provider)))

(defn access-token-req-base
  "Base request map for access-token requests."
  [oauth2-provider]
  (let [{:keys [access-token-url]} (oauth2-spec oauth2-provider)]
    {:url access-token-url
     :method :post
     :accept :json
     :as :json
     :form-params (-> oauth2-provider
                      oauth2-param-gen
                      base-grant-params)}))

(defn authorization-code-req
  "Creates a request for an OAuth2 access token using the authorization_code
  grant type"
  [oauth2-provider code state code-verifier]
  (update (access-token-req-base oauth2-provider)
          :form-params
          merge
          {;; for verification by GitHub only
           :redirect_uri (callback-url oauth2-provider)
           :state state
           :code_verifier code-verifier}
          (-> oauth2-provider
              oauth2-param-gen
              (code-grant-params code))))

(defn client-credentials-req
  "Creates a request for an OAuth2 access token using the client_credentials
  grant type. OAuth2 consumer has to be marked private (in BitBucket) for this to work."
  [oauth2-provider]
  (update (access-token-req-base oauth2-provider)
          :form-params
          merge
          (-> oauth2-provider
              oauth2-param-gen
              client-credentials-params)))

(defn refresh-token-req
  "Creates a request for an OAuth2 access token using the refresh_token
  grant type."
  [oauth2-provider refresh-token]
  (update (access-token-req-base oauth2-provider)
          :form-params
          merge
          {:redirect_uri (callback-url oauth2-provider)}
          (-> oauth2-provider
              oauth2-param-gen
              (refresh-grant-params refresh-token))))

(defn provider-authorize-url
  "Returns authorize URL of a provider complete with all the parameters.

  Params contains keys for determination of the scope string, likely from
  the request params."
  [oauth2-provider params state code-verifier]
  (->> {;; https://docs.github.com/en/developers/apps/building-oauth-apps/authorizing-oauth-apps#redirect-urls
        :redirect_uri (callback-url oauth2-provider)
        :scope (oauth2-scope oauth2-provider params)
        :state state
        :code_challenge (data/sha-256-base64 code-verifier)
        :code_challenge_method "S256"}
       (merge (-> oauth2-provider oauth2-param-gen authorize-params))
       (url-utils/add-params (:authorize-url (oauth2-spec oauth2-provider)))))

(defn oauth2-authorize-url
  "Return url for redirect to start the OAuth flow, that is a URL pointing at our OAuth2 endpoint.

   If OAuth2 provider requires an https callback URL, then we redirect user to public version of the site.

  In production systems base-uri and public-base-uri should be one and the same,
  but in development we have a bit of a problem: the host of the callback must match the current host.

  This is important:
  If you are on http://localhost and your callback is on https://ngrok... then you have
  a different session when the provider redirects back and the 'state' parameter will be absent
  from session!!!

  We ask for base-url from oauth2-provider, so we can try to match it at the start of the flow."
  [oauth2-provider next-uri additional-params]
  (let [[base-url relative-uri] (oauth2-mount-point oauth2-provider)]
    (-> (str base-url relative-uri "/authorize")
        (url-utils/add-params (assoc additional-params :next (or (url-utils/try-relative-url next-uri) "/"))))))
