(ns codescene.features.client.api
  "A collection of utilities for common tasks interacting with integration APIs.

  Often the code is very similar between different integrations, but not identical,
  so it is not possible to write complete, generic workflows, but some code can be reused
  and it's stored here.

  The API functions require an object satisfying ApiClient protocol (which augments
  requests with correct authentication headers).
  This protocol is implemented for nil and String as well for the purposes of REPL use.

  You can use nil authentication token to call an API function without any authentication.
  You can use a String containing an access token to call an API function as that user without any
  refresh token logic, retries, token storing/removing etc...

  When an expired token response is encountered, the ApiClient object is asked to provide an
  alternative non-nil ApiClient object (technically it can return itself, though),
  throwing the expired token exception if nil is returned. That ApiClient is then used to
  try the request again, recursively.

  All the client code must be coded to respect the above rules.

  This is used for Refresh token handling: if access token is expired, a new one is fetched via
  refresh token, and that String is returned. String is a valid ApiClient, that doesn't implement
  any refreshing strategy, so only 1 retry is done.

  RenewableAccessToken ==401==> String (raw access token) ==401==> nil (no retry at this point)"
  (:require [clj-http.client :as http-client]
            [codescene.features.client.oauth2 :as oauth2]
            [codescene.features.components.http :as f.http]
            [codescene.features.util.string :as u.str]
            [codescene.features.util.url :as u.url]
            [codescene.util.json :as json]
            [codescene.util.async :as async]
            [evolutionary-metrics.mining.file-patterns :as file-patterns]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [taoensso.timbre :as log]
            [ring.util.response :refer [find-header]])
  (:import (clojure.lang IPersistentVector)
           (java.util.concurrent ForkJoinPool)))

;; TODO ROK API TODO LIST
;; specs on all functions
;; split provider functions into multiple namespaces

(defn check-interrupt []
  (when (Thread/interrupted)
    (throw (InterruptedException.))))

(defprotocol ApiClient
  "Object that adds authentication and other stateful properties (such as connection manager) to a request map."
  (augment-request [this req] "Returns request with additional security info, can return nil to abort.")
  (on-expired-token [this] "This is called when a 401 response because of expired token is encountered.
  If this returns a non-nil ApiClient the request will be retried with those credentials.
  If an exception is thrown, that exception will be logged and swallowed.")
  )

(defn json-safe-parse [string]
  (try
    (json/parse-string string)
    (catch Exception _ nil)))

(defn limit-items
  "Modifies add-page function to prevent further paging if number of loaded items matches or exceeds max-items"
  [paging-state max-items]
  (let [limiter #(if (>= (count (:items % [])) max-items) (assoc % :cursor nil) %)]
    (update paging-state :add-page #(comp limiter %))))

(defn with-item-xf
  "Modifies add-page function to apply transducer to page items"
  [paging-state xf]
  (update paging-state :add-page #(fn [page & more] (apply % (into [] xf page) more))))

(defn printable-req [req]
  (let [truncate #(u.str/hide % 6)]
    (-> req
        (dissoc :connection-manager :max-leased :async-conn-manager)
        (m/update-existing :oauth-token truncate)
        (m/update-existing :basic-auth (partial mapv truncate))
        (m/update-existing-in [:headers "authorization"] truncate)
        (m/update-existing-in [:headers "PRIVATE-TOKEN"] truncate))))

(extend-protocol ApiClient
  nil
  (augment-request [this req] req)
  (on-expired-token [this] nil)
  String
  (augment-request [this req]
    (if (find-header req "authorization")
      req
      (assoc req :oauth-token this)))
  (on-expired-token [this] nil)
  IPersistentVector
  (augment-request [this req]
    (if (find-header req "authorization")
      req
      (assoc req :basic-auth this)))
  (on-expired-token [this] nil))

;; add Authentication header "Token ...."
(defrecord TokenAuth [token]
  ApiClient
  (augment-request [this req]
    (if (find-header req "authorization")
      req
      (assoc-in req [:headers "authorization"] (str "Token " token))))
  (on-expired-token [this] nil))

;; RenewableAccessToken encapsulates access-token, refresh-token pair
;; in a mutating state. This token performs actual renew token calls to provider.
(defrecord RenewableAccessToken [access-token refresh-token oauth2-provider http-client-config]
  ApiClient
  (augment-request [this req] (augment-request @access-token req))
  (on-expired-token [this]
    (when @refresh-token
      (log/debugf "RenewableAccessToken starting renew on %s" oauth2-provider)
      (let [{:keys [refresh_token access_token]}
            (->> (oauth2/refresh-token-req oauth2-provider @refresh-token)
                 (meta-merge http-client-config {:as :json})
                 http-client/request
                 :body)]
        ;; This returns a String access token, which implements ApiClient,
        ;; but it doesn't "renew", so it results in only 1 retry.
        (reset! refresh-token refresh_token)
        (reset! access-token access_token)))))

(defn retry-token
  "Checks if exception fits the retry criteria, returns token for request retry
  or throws the exception e.g. if the retry function returned nil."
  [e auth-token]
  (or (and (:token-expired? (ex-data e))
           (try (on-expired-token auth-token)
                (catch Exception e2
                  (log/warnf e2 "on-expired-token threw an exception"))))
      (throw e)))

(defrecord ExtraProperties [prop-map delegate]
  ApiClient
  (augment-request [this req] (meta-merge prop-map (augment-request delegate req)))
  (on-expired-token [this] (when-let [new-delegate (on-expired-token delegate)]
                             (->ExtraProperties prop-map new-delegate))))

(defmacro to-url
  "Concatenates base plus parts using / into a path.

  Values are treated as such:
  - base value has any trailing slashes removed
  - string parts have slashes trimmed on both sides, except if last part is string,
  then it only has slash trimmed on left
  - other parts (such as variables) are wrapped in a string conversion and
  url encode (which encodes characters invalid in paths).

  e.g. (to-url 'https://google.com/' x '/x/') where x is '//' is 'https://google.com/%2F%2F/x/'"
  [base & parts]
  `(u.url/make-url ~base ~@parts))

(defn fix-api-url [{:keys [api-url] :as params} default-api-url]
  (let [api-url (if (seq api-url) api-url default-api-url)
        api-url (some-> api-url (file-patterns/trimr-ch \/))]
    (update params :url #(if (re-find #"^https?://" %)
                           %
                           (str api-url \/ (file-patterns/triml-ch % \/))))))

(defn fix-connection-manager
  "Clj-http requires a different kind of connection-manager if doing async, they are of
  totally different types. Here we fix this issue, by pulling in a different CM when async.

  Even if async CM isn't defined we need to use nil, as they sync one doesn't work at all."
  [params]
  (cond-> params
    (:async? params) (assoc :connection-manager (:async-conn-manager params)
                            :http-client (:async-http-client params))))

(defrecord PropUpdatingClient [delegate prop f]
  ApiClient
  (augment-request [this req]
    (update (augment-request delegate req) prop f))
  (on-expired-token [this]
    (when-let [new-delegate (on-expired-token delegate)]
      (->PropUpdatingClient new-delegate prop f))))

(defn with-prefix
  "Create a client that prefixes request URLs with a certain URL, unless they are absolute URL"
  [api-client prefix]
  (->PropUpdatingClient api-client
                        :url
                        #(if (re-find #"^https?://" %)
                           %
                           (str prefix \/ (file-patterns/triml-ch % \/)))))

(defn with-updated-api-url
  "Runs the update function on api-url"
  [api-client api-url-update-fn]
  (->PropUpdatingClient api-client
                        :api-url
                        #(when % (api-url-update-fn %))))

(defn ->authed-client
  ([system auth-token] (->authed-client system auth-token nil nil))
  ([system auth-token api-url] (->authed-client system auth-token api-url nil))
  ([system auth-token api-url timeout]
   (->ExtraProperties
     (m/assoc-some (f.http/client-config system)
       :api-url api-url
       :socket-timeout timeout
       :connection-timeout timeout)
     auth-token)))

(defn req-fn
  "Creates a request function that takes (fn [authed-client method url request-params])

  - always-params is clj-http options that are added to all requests
  - default-url is the URL that is prepended to all requests that do not use absolute URL and have no api-url option
  - do-request-fn is (fn [req error-handler] ...), it should run (error-handler e) on errors thrown, see do-request in this namespace
  - process-exception is a function (fn [e final-params]) that handles exception and returns a new exception, that will get thrown

  There are additional parameters in http request that control the execution:
  - api-url: if provided, that will prefix all request URLs, otherwise default-url parameter is used"
  [always-opts default-url process-exception]
  (fn do-request
    [authed-client method url request-params]
    (let [params (meta-merge
                   {:url url
                    :request-method method}
                   always-opts
                   request-params)
          final-params (-> authed-client
                           (augment-request params)
                           (fix-api-url default-url)
                           fix-connection-manager)
          err-handler (bound-fn err-handler [e]
                        (-> e
                            (process-exception final-params)
                            (retry-token authed-client)
                            (do-request method url request-params)))]
      (when-not (:connection-manager final-params)
        (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
      (if (:async? final-params)
        (-> (partial http-client/request final-params)
            async/cb->promise
            ;; don't run processing in IO reactor thread
            (async/handle (fn [result e] (if e (err-handler e) result)) (ForkJoinPool/commonPool)))
        (try
          (http-client/request final-params)
          (catch Exception e
            (err-handler e)))))))

(defn body-fn [req-fn] (comp #(async/then % :body) req-fn))


(defn service-exception
  [service cause extra]
  (let [obj (merge (when (map? cause) cause)
                   {:service service}
                   extra)
        msg (condp instance? cause
              String cause
              Throwable (ex-message cause)
              (pr-str obj))]
    (ex-info (or (:message obj) msg)
             (merge {:message msg} obj)
             (when (instance? Throwable cause) cause))))

(defn ex-http-error
  ([service cause status]
   (ex-http-error service cause status {}))
  ([service cause status extra]
   (service-exception service cause (merge {:type :http-error :status status} extra))))

(defn ex-unauthorized
  "An exception that represents an error with the token,
   which can be resolved by relogging,
   which can be either wrong, expired token or no token at all or
   problem with scope."
  ([service cause] (ex-unauthorized service cause {}))
  ([service cause extra] (ex-http-error service cause 401 (merge extra {:auth-required? true}))))

(defn ex-forbidden
  "An exception that represents unauthorized access.

   Unauthorized access happens when there's a problem with user's access
   token (lack of rights) that cannot be resolved by relogging."
  [service cause extra]
  (ex-http-error service cause 403 extra))